<?php
namespace PowerElements;
use PowerElements\Helpers\Utils as Utils;

use \Elementor\Repeater;
use \Elementor\Widget_Base;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Scheme_Typography;
use \Elementor\Group_Control_Text_Shadow;


	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class PowerElements_Team extends Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_elements_team';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Team', 'power-elements' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-lock-user';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'peeao_team_contenttab',
				[
					'label' => __( 'Team Content', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'peeao_team_style',
				[
					'label'   => __( 'Style', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'style_1',
					'options' => [
						'style_1' => __( 'Style 1', 'power-elements' ),
						'style_2' => __( 'Style 2', 'power-elements' ),
						'style_3' => __( 'Style 3', 'power-elements' ),
						'style_4' => __( 'Style 4', 'power-elements' )
					]
				]
			);
			$this->add_control(
				'peeao_team_name_text',
				[
					'label'       => __( 'Name', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'Anders Angstrom', 'power-elements' ),
					'placeholder' => __( 'Your Name Please', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_designation_text',
				[
					'label'       => __( 'Address or Designation', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'CEO, Advertising Corp', 'power-elements' ),
					'placeholder' => __( 'Give your designation please.', 'power-elements' ),
					'label_block' => true
				]
			);
			$this->add_control(
				'peeao_team_description_text',
				[
					'label'       => __( 'Description', 'power-elements' ),
					'type'        => Controls_Manager::TEXTAREA,
					'default'     => __( '', 'power-elements' ),
					'placeholder' => __( 'Short description.', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_image',
				[
					'label'   => __( 'Choose Image', 'power-elements' ),
					'type'    => Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			$this->add_group_control(
				Group_Control_Image_Size::get_type(),
				[
					'name' => 'thumbnail',
					'default' => 'full',
				]
			);
			$this->add_control(
				'peeao_team_social_options',
				[
					'label'   => __( 'Social Link', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'hide',
					'options' => [
						'show' => __( 'Show', 'power-elements' ),
						'hide' => __( 'Hide', 'power-elements' )
					]
				]
			);
			$this->add_control(
				'peeao_team_social_options_display_position',
				[
					'label'   => __( 'Style', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'top',
					'options' => [
						'top' => __( 'Top', 'power-elements' ),
						'bottom' => __( 'Bottom', 'power-elements' ),
					]
				]
			);
			$repeater = new Repeater();
			$repeater->add_control(
				'peeao_team_social_icon',
				[
					'label' => esc_html__( 'Social Icons', 'power-elements' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => 'fa fa-facebook',
					],
				]
			);
			$repeater->add_control(
				'peeao_team_social_link',
				[
					'label'         => __( 'Link', 'power-elements' ),
					'type'          => Controls_Manager::URL,
					'placeholder'   => __( 'https://your-link.com', 'power-elements' ),
					'show_external' => true,
					'default'       => [
						'url'         => '',
						'is_external' => true,
						'nofollow'    => true,
					]
				]
			); 
			$this->add_control(
				'peeao_team_list',
				[
					'label'       => __( 'Social List', 'power-elements' ),
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'peeao_team_social_link' =>  '#' ,
							'peeao_team_social_icon' => 'fa fa-facebook',
						],
						[
							'peeao_team_social_link' => '#',
							'peeao_team_social_icon' => 'fa fa-twitter',
						],
						[
							'peeao_team_social_link' => '#',
							'peeao_team_social_icon' => 'fa fa-instagram',
						],
					],
					'condition'   => [
						'peeao_team_social_options' => 'show',
					]
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'peeao_team_style_tab',
				[
					'label' => __( 'Image', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_img_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member_image',
				]
			);
			$this->add_responsive_control(
				'peeao_team_img_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_img_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_more_options',
				[
					'label' => __( 'Overlay', 'power-elements' ),
					'type' => Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'     => 'peeao_team_img_overlay_background',
					'label'    => __( 'Background Overlay', 'power-elements' ),
					'types'    => [ 'classic', 'gradient'],
					'show_label'=>true,
					'selector' => '{{WRAPPER}} .power_team_member.simple_style .power_team_member_image:before',
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'peeao_team_intro_style',
				[
					'label' => __( 'Intro', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'peeao_team_intro_align',
				[
					'label'        => __( 'Alignment', 'power-elements' ),
					'type'         => Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __( 'Left', 'power-elements' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power-elements' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power-elements' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power-elements' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'default'  => 'center',
					'selectors'  => [
						'{{WRAPPER}} .power_team_member.overlay_style .power_team_member_intro, {{WRAPPER}} .power_team_member.simple_style .power_team_member_details' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'     => 'peeao_team_intro_background',
					'label'    => __( 'Background', 'power-elements' ),
					'types'    => [ 'classic', 'gradient'],
					'show_label'=>true,
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_intro, {{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner',
				]
			);
			$this->add_responsive_control(
				'peeao_team_intro_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_intro' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_intro_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_intro, {{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'peeao_team_details_style',
				[
					'label' => __( 'Content Wrapper', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'peeao_team_details_align',
				[
					'label'        => __( 'Alignment', 'power-elements' ),
					'type'         => Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __( 'Left', 'power-elements' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power-elements' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power-elements' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power-elements' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details' => 'text-align: {{VALUE}};',
						'{{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'     => 'peeao_team_details_background',
					'label'    => __( 'Background', 'power-elements' ),
					'types'    => [ 'classic', 'gradient'],
					'show_label'=>true,
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_details, {{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner',
				]
			);
			$this->add_responsive_control(
				'peeao_team_details_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_details_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details, {{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Name Style
			 */
			$this->start_controls_section(
				'peeao_team_name_style',
				[
					'label' => __( 'Name', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_team_title_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_title',
				]
			);
			$this->start_controls_tabs( 'peeao_team_tabs_style' );
			
			$this->start_controls_tab(
				'peeao_team_tab_style_normal',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_name_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'  => '#222222',
					'selectors' => [
						'{{WRAPPER}} .power_team_member_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->end_controls_tab();
			
			$this->start_controls_tab(
				'peeao_team_tab_style_before',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
			
			$this->add_control(
				'peeao_team_hover_text_color',
				[
					'label'     => __( 'Text Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member:hover .power_team_member_inner .power_team_member_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();

			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_team_name_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member_title',
				]
			);
			$this->add_responsive_control(
				'peeao_team_name_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_name_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Designation Style
			 */
			$this->start_controls_section(
				'peeao_team_designation_style',
				[
					'label' => __( 'Designation', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_team_designation_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'fields_options' => [
						'font_weight' => [
							'default' => '700',
						],
						'font_family' => [
							'default' => 'Hind',
						],
						'font_size'   => [
							'default' => [ 
								'unit' => 'px', 
								'size' => 12 
							]
						],
						'text_transform' => [
							'default' => 'uppercase'
						]
					],
					'selector' => '{{WRAPPER}} .power_team_member_designation',
				]
			);
			$this->add_control(
				'peeao_team_designation_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'  => '#222222',
					'selectors' => [
						'{{WRAPPER}} .power_team_member_designation' => 'color: {{VALUE}} !important',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_team_designation_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member_designation',
				]
			);
			$this->add_responsive_control(
				'peeao_team_designation_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_designation' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_designation_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_designation' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Description Style
			 */
			$this->start_controls_section(
				'peeao_team_des_style',
				[
					'label' => __( 'Description', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_team_des_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'peeao_team_des_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_text' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_team_des_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_responsive_control(
				'peeao_team_des_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_des_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
			/*
			 * Social Style
			 */
			$this->start_controls_section(
				'peeao_team_social_style',
				[
					'label' => __( 'Social', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_align',
				[
					'label'        => __( 'Alignment', 'power-elements' ),
					'type'         => Controls_Manager::CHOOSE,
					'options'      => [
						'flex-start'    => [
							'title' => __( 'Left', 'power-elements' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power-elements' ),
							'icon'  => 'fa fa-align-center',
						],
						'flex-end'   => [
							'title' => __( 'Right', 'power-elements' ),
							'icon'  => 'fa fa-align-right',
						]
					],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social' => 'align-items: {{VALUE}}; justify-content:{{VALUE}};',
					],
				]
			);
			$this->start_controls_tabs(
				'peeao_team_style_social_tabs'
			);
			/*
			 *
			 * SOCIAL NORMAL
			 */
			$this->start_controls_tab(
				'peeao_team_style_social_normal_tab',
				[
					'label' => __( 'Social Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_social_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_social_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_width',
				[
					'label'     => __( 'Icon Width', 'power-elements' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 40,
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_height',
				[
					'label'     => __( 'Icon Height', 'power-elements' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 40,
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_lineheight',
				[
					'label'     => __( 'Line Height', 'power-elements' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 40,
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'line-height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_social_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a',
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 *
			 * SOCIAL HOVER
			 */
			$this->start_controls_tab(
				'style_social_hover_tab',
				[
					'label' => __( 'Social Hover', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_social_hover_color',
				[
					'label'     => __( 'Hover Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_social_bg_hover_color',
				[
					'label'     => __( 'Background Hover Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_social_hover_border',
					'label'    => __( 'Hover Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover',
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_hover_radius',
				[
					'label'      => __( 'Border Hover Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'peeao_team_hr-tabs-end',
				[
					'type' => Controls_Manager::DIVIDER,
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_social_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings();


			switch($settings['peeao_team_style']){
				case 'style_1':
					$klass = 'simple_style'; break;
				case 'style_2':
					$klass = 'hover_overlay_style'; break; 
				case 'style_3':
					$klass = 'hover_overlay_style'; break;
				case 'style_4':
					$klass = 'overlay_style'; break;
				default:
					$klass = 'simple_style';
			}
			if(isset($settings['peeao_team_list'])){
				foreach ( $settings['peeao_team_list'] as $item ) {
					$social_li = '<li class="power_team_member_social_item">
									<a href="' . esc_url( $item['peeao_team_social_link']['url'] ) . '"><i class="' . esc_attr( $item['peeao_team_social_icon']['value'] ) . '"></i></a>
								</li>';
				}
			}else{
				$social_li = '';
			}

			$social = $settings['peeao_team_social_options'] == 'show' ? 
						'<ul class="power_team_member_social">'.$social_li.'</ul>' : '';

			$img = $settings['peeao_team_image']['url'] != '' ?
				'<div class="power_team_member_image">
				'.Group_Control_Image_Size::get_attachment_image_html( $settings, 'thumbnail', 'peeao_team_image' ).'
				</div>' : '';

			$description = $settings['peeao_team_description_text'] != '' ? 
				'<p class="power_team_member_text">' . $settings['peeao_team_description_text'] . '</p>' : '';

			$intro_html = '<div class="power_team_member_intro">
								<h4 class="power_team_member_intro_title">' . esc_html( $settings['peeao_team_name_text'] ) . '</h3>
								<h6 class="power_team_member_intro_designation">' . esc_html( $settings['peeao_team_designation_text'] ) . '</h5>
							</div>'; ?>

			<div class="peeao-wrapper">
				<div class="power_team_member <?php echo esc_attr($klass); ?> ">
					<?php echo Utils::kses($img); ?> 
					<?php if ( $settings['peeao_team_style'] == 'style_2' ) {
						echo Utils::kses($intro_html);
					} ?>
					<div class="power_team_member_details">
						<div class="power_team_member_inner">
							<?php if ( $settings['peeao_team_social_options_display_position'] == 'top' ) { 
								echo  Utils::kses($social); 
							} ?>
							<h3 class="power_team_member_title"><?php echo  Utils::kses( $settings['peeao_team_name_text'] ) ; ?></h3>
							<h5 class="power_team_member_designation"><?php echo  Utils::kses( $settings['peeao_team_designation_text'] ) ; ?></h5>
							<?php echo  Utils::kses($description); ?>
							<?php if ( $settings['peeao_team_social_options_display_position'] == 'bottom' ) { 
								echo  Utils::kses($social); 
							} ?>
						</div>
					</div>
				</div>
			</div>
		<?php }
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}