<?php
namespace PowerElements;
use PowerElements\Helpers\Utils as Utils;

use \Elementor\Repeater;
use \Elementor\Widget_Base;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;


	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class PowerElements_Team_Carousel extends Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_elements_team_carousel';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Team Carousel', 'power-elements' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-lock-user';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'peeao_team_carousel_content',
				[
					'label' => __( 'Team Content', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'peeao_team_carousel_style',
				[
					'label'   => __( 'Style', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'style_1',
					'options' => [
						'style_1' => __( 'Style 1', 'power-elements' ),
						'style_2' => __( 'Style 2', 'power-elements' ),
						'style_3' => __( 'Style 3', 'power-elements' ),
						'style_4' => __( 'Style 4', 'power-elements' )
					]
				]
			);
			$repeater = new Repeater();
			$repeater->add_control(
				'name_text',
				[
					'label'       => __( 'Name', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'Your Name', 'power-elements' ),
					'placeholder' => __( 'Your Name Please', 'power-elements' ),
				]
			);
			$repeater->add_control(
				'designation_text',
				[
					'label'       => __( 'Address or Designation', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'CEO, Advertising Corp', 'power-elements' ),
					'placeholder' => __( 'Give your designation please.', 'power-elements' ),
					'label_block' => true
				]
			);
			$repeater->add_control(
				'description_text',
				[
					'label'       => __( 'Description', 'power-elements' ),
					'type'        => Controls_Manager::TEXTAREA,
					'default'     => __( '', 'power-elements' ),
					'placeholder' => __( 'Short description.', 'power-elements' ),
				]
			);
			$repeater->add_control(
				'image',
				[
					'label'   => __( 'Choose Image', 'power-elements' ),
					'type'    => Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			$repeater->add_group_control(
				Group_Control_Image_Size::get_type(),
				[
					'name' => 'thumbnail',
					'default' => 'full',
				]
			);
			$this->add_control(
				'peeao_team_carousel_list',
				[
					'label'       => __( 'Team List', 'power-elements' ),
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'name_text'   => __( 'Anders Angstrom', 'power-elements' ),
							'description_text' =>'',
							'designation_text' => __( 'Team Member', 'power-elements' ),
						],
						[
							'name_text'   => __( 'Anders Angstrom', 'power-elements' ),
							'description_text' => '',
							'designation_text' => __( 'Team Member', 'power-elements' ),
						],
						[
							'name_text'   => __( 'Anders Angstrom', 'power-elements' ),
							'description_text' => '',
							'designation_text' => __( 'Team Member', 'power-elements' ),
						],
						[
							'name_text'   => __( 'Anders Angstrom', 'power-elements' ),
							'description_text' => '',
							'designation_text' => __( 'Team Member', 'power-elements' ),
						],
						[
							'name_text'   => __( 'Anders Angstrom', 'power-elements' ),
							'description_text' => '',
							'designation_text' => __( 'Team Member', 'power-elements' ),
						],
					],
					'title_field' => '{{{ name_text }}}',
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL SETTINGS
			 */
			$this->start_controls_section(
				'section_team_carousel_settings',
				[
					'label' => __( 'Settings', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'peeao_team_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power-elements' ),
					'type'        => Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 4,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_item_mobile',
				[
					'label'       => __( 'Mobile Item', 'power-elements' ),
					'type'        => Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_margin',
				[
					'label'   => __( 'Margin', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_padding',
				[
					'label'   => __( 'Padding', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_responsive_class',
				[
					'label'   => __( 'Responsive Class', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots',
				[
					'label'   => __( 'Dots', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_left_icon',
				[
					'label' => esc_html__( 'Left Nav Icon', 'power-elements' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => '',
						'library' => 'peeaoicons',
					],
					'condition'   => [
						'peeao_team_carousel_nav' => 'true',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_right_icon',
				[
					'label' => esc_html__( 'Right Nav Icon', 'power-elements' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => '',
						'library' => 'peeaoicons',
					],
					'condition'   => [
						'peeao_team_carousel_nav' => 'true',
					],
				]
			);
			$this->end_controls_section(); 
			/*
			 * STYLE
			 */
			$this->start_controls_section(
				'peeao_team_carousel_style_tab',
				[
					'label' => __( 'Image', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_carousel_img_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_image',
				]
			);
			$this->add_control(
				'peeao_team_carousel_img_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_img_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_more_options',
				[
					'label'     => __( 'Overlay', 'power-elements' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_team_carousel_img_overlay_background',
					'label'      => __( 'Background Overlay', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel.simple_style .power_team_member_carousel_image:before',
				]
			);
			$this->end_controls_section();
			/*
			 *
			 * DETAILS STYLE
			 */
			
			$this->start_controls_section(
				'peeao_team_carousel_content_wrapper_style_tab',
				[
					'label' => __( 'Content Wrapper', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'peeao_team_carousel_details_align',
				[
					'label'     => __( 'Alignment', 'power-elements' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => __( 'Left', 'power-elements' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power-elements' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power-elements' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power-elements' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'default'  => 'center',
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel .power_team_member_carousel_details'                                        => 'text-align: {{VALUE}};',
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel.hover_overlay_style .power_team_member_carousel_intro'                            => 'text-align: {{VALUE}};',
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel.hover_overlay_style .power_team_member_carousel_details .power_team_member_carousel_inner' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_team_carousel_details_background',
					'label'      => __( 'Background', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel .power_team_member_carousel_details, {{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel.hover_overlay_style .power_team_member_carousel_details .power_team_member_carousel_inner',
				]
			);
			$this->add_control(
				'peeao_team_carousel_details_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel .power_team_member_carousel_details' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_details_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel .power_team_member_carousel_details, {{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel.hover_overlay_style .power_team_member_carousel_details .power_team_member_carousel_inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Name Style
			 */
			$this->start_controls_section(
				'peeao_team_carousel_name_style_tab',
				[
					'label' => __( 'Name', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_team_carousel_title_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_title',
				]
			);
			$this->add_control(
				'peeao_team_carousel_name_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'  => '#222222',
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_name_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_title',
				]
			);
			$this->add_control(
				'peeao_team_carousel_name_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_name_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Designation Style
			 */
			$this->start_controls_section(
				'peeao_team_carousel_designation_style_tab',
				[
					'label' => __( 'Designation', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_team_carousel_designation_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_designation',
				]
			);
			$this->add_control(
				'peeao_team_carousel_designation_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'  => '#222222',
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_designation' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_designation_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_team_member_carousel_designation',
				]
			);
			$this->add_control(
				'peeao_team_carousel_designation_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_designation' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_designation_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_designation' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Description Style
			 */
			$this->start_controls_section(
				'peeao_team_carousel_des_style_tab',
				[
					'label' => __( 'Description', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_team_carousel_des_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_text',
				]
			);
			$this->add_control(
				'peeao_team_carousel_des_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_text' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_des_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_text',
				]
			);
			$this->add_control(
				'peeao_team_carousel_des_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_des_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel .power_team_member_carousel_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();



			/*
			 * READ MORE Style
			 */
			$this->start_controls_section(
				'peeao_team_carousel_nav_style_tab',
				[
					'label'     => __( 'Nav', 'power-elements' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'peeao_team_carousel_nav' => 'true',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_carousel_nav_prev_spacing',
				[
					'label'   => __( 'Previous arrow spacing', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 40,
						'max'  => 200,
						'step' => 1,
					],
					'default' => [
						'size' => 40,
					],
					'condition'   => [
						'peeao_team_carousel_nav_style' => 'power_owl_nav_side',
					],
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev' => 'left: -{{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_team_carousel_nav_next_spacing',
				[
					'label'   => __( 'Next arrow spacing', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 40,
						'max'  => 200,
						'step' => 1,
					],
					'default' => [
						'size' => 40,
					],
					'condition'   => [
						'peeao_team_carousel_nav_style' => 'power_owl_nav_side',
					],
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next' => 'right: -{{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->start_controls_tabs(
				'peeao_team_carousel_nav_tabs'
			);
			
			$this->start_controls_tab(
				'peeao_team_carousel_nav_normal_tab',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev' => 'color: {{VALUE}}',
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next' => 'color: {{VALUE}}',
						'{{WRAPPER}} .power_outside_grid_owl_prev, .power_outside_grid_owl_next' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev' => 'background-color: {{VALUE}}',
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next' => 'background-color: {{VALUE}}',
						'{{WRAPPER}} .power_outside_grid_owl_prev, .power_outside_grid_owl_next' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_nav_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next, {{WRAPPER}} .power_outside_grid_owl_next,{{WRAPPER}} .power_outside_grid_owl_prev',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_carousel_nav_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next, {{WRAPPER}} .power_outside_grid_owl_next,{{WRAPPER}} .power_outside_grid_owl_prev',
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
					],
				]
			);
			$this->end_controls_tab();
			
			$this->start_controls_tab(
				'peeao_team_carousel_nav_hover_tab',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_hover_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover ,{{WRAPPER}} .power_outside_grid_owl_prev:hover' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_bg_hover_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_nav_hover_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_carousel_nav_hover_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover',
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_hover_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_nav_hover_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_teamcarousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
					],
				]
			);
			$this->end_controls_tab();
			
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * DOTS Style
			 */
			$this->start_controls_section(
				'peeao_team_carousel_dots_style_tab',
				[
					'label'     => __( 'Dots', 'power-elements' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'peeao_team_carousel_dots' => 'true',
					],
				]
			);
			$this->start_controls_tabs(
				'peeao_team_carousel_dots_tabs'
			);
			
			$this->start_controls_tab(
				'peeao_team_carousel_dots_normal_tab',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot span' => 'background: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_dots_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_carousel_dots_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot',
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'peeao_team_carousel_dots_hover_tab',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_hover_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot:hover span' => 'background: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_hover_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_dots_hover_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot:hover',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_carousel_dots_hover_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot:hover',
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_hover_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'peeao_team_carousel_dots_active_tab',
				[
					'label' => __( 'Active', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_active_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot.active span' => 'background: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_active_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot.active' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_team_carousel_dots_active_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot.active',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_team_carousel_dots_active_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot.active',
				]
			);
			$this->add_control(
				'peeao_team_carousel_dots_active_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_teamcarousel.owl-theme .owl-dots .owl-dot.active' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			
			$this->end_controls_tabs();
			$this->end_controls_section();
		}
		
		 
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings();


			switch($settings['peeao_team_carousel_style']){
				case 'style_1':
					$klass = 'simple_style'; break;
				case 'style_2':
					$klass = 'hover_overlay_style'; break;
				case 'style_3':
					$klass = 'hover_overlay_style'; break;
				case 'style_4':
					$klass = 'overlay_style'; break;
				default:
					$klass = 'simple_style';
			} ?>

			
			<div class="peeao-wrapper">
				<?php if ( $settings['peeao_team_carousel_list'] ) {
					echo '<div class="owl-carousel owl-theme power_teamcarousel"  data-owl-carousel=\'{"items": ' . esc_attr( $settings['peeao_team_carousel_item'] ) . ', "margin": ' . esc_attr( $settings['peeao_team_carousel_margin']['size'] ) . ', "padding": ' . esc_attr( $settings['peeao_team_carousel_padding']['size'] ) . ',  "autoplay": ' . esc_attr( $settings['peeao_team_carousel_autoplay'] ) . ',  "responsiveClass": ' . esc_attr( $settings['peeao_team_carousel_responsive_class'] ) . ',  "dots": ' . esc_attr( $settings['peeao_team_carousel_dots'] ) . ', "nav": ' . esc_attr( $settings['peeao_team_carousel_nav'] ) . ',"responsive":{ "0" :{ "items": ' . esc_attr( $settings['peeao_team_carousel_item_mobile'] ) . ' }, "540" :{ "items": ' . esc_attr( $settings['peeao_team_carousel_item_mobile'] ) . ' }, "860" :{ "items" : ' . esc_attr( $settings['peeao_team_carousel_item'] ) . ' } , "1000":{ "items" : ' . esc_attr( $settings['peeao_team_carousel_item'] ) . ' }}}\'>';
						foreach ( $settings['peeao_team_carousel_list'] as $item ) {
							$img = $item['image']['url'] != '' ?
								'<div class="power_team_member_carousel_image">
								'.Group_Control_Image_Size::get_attachment_image_html( $item, 'thumbnail', 'image' ).'
								</div>' : '';
							$description = $item['description_text'] != '' ? 
								'<p class="power_team_member_carousel_text">' . $item['description_text'] . '</p>' : ''; 
								
							$intro_html = '<div class="power_team_member_carousel_intro">
									<h4 class="power_team_member_carousel_intro_title">' . esc_html( $item['name_text'] ) . '</h3>
									<h6 class="power_team_member_carousel_intro_designation">' . esc_html( $item['designation_text'] ) . '</h5>
								</div>'; ?> 
 
							<div class="item">
								<div class="power_team_member_carousel <?php echo esc_attr($klass); ?> ">
									<?php echo Utils::kses($img); ?> 
									<?php if ( $settings['peeao_team_carousel_style'] == 'style_2' ) {
										echo Utils::kses($intro_html);
									} ?>
									<div class="power_team_member_carousel_details">
										<div class="power_team_member_carousel_inner">
											<h3 class="power_team_member_carousel_title"><?php echo  Utils::kses( $item['name_text'] ) ; ?></h3>
											<h5 class="power_team_member_carousel_designation"><?php echo  Utils::kses( $item['designation_text'] ) ; ?></h5>
											<?php echo Utils::kses($description); ?>
										</div>
									</div>
								</div>
							</div>
						<?php } ?>
					</div>
				<?php } else {
					echo esc_html__( 'Something wrong', 'power-elements' );
				} ?>
			</div>
		<?php }


		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}