<?php
namespace PowerElements;

use \Elementor\Repeater;
use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Scheme_Typography;
use \Elementor\Group_Control_Text_Shadow;


	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class PowerElements_Slider extends Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_elements_slider';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Slider', 'power-elements' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-media-carousel';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'peeao_slider_content',
				[
					'label' => __( 'Slider Content', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$repeater = new Repeater();
			$repeater->add_control(
				'sub_heading',
				[
					'label'       => __( 'Sub Heading', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'Welcome', 'power-elements' ),
					'placeholder' => __( '', 'power-elements' ),
				]
			);
			$repeater->add_control(
				'heading',
				[
					'label'       => __( 'Heading', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'Creative design and advertising', 'power-elements' ),
					'placeholder' => __( '', 'power-elements' ),
					'label_block' => true
				]
			);
			$repeater->add_control(
				'description_text',
				[
					'label'       => __( 'Description', 'power-elements' ),
					'type'        => Controls_Manager::TEXTAREA,
					'default'     => __( 'Eos an odio detracto, per ne aliquam vituperata. Sit ea feugiat fuisset.', 'power-elements' ),
					'placeholder' => __( '', 'power-elements' ),
				]
			);
			$repeater->add_control(
				'peeao_slider_image',
				[
					'label'   => __( 'Choose Image', 'power-elements' ),
					'type'    => Controls_Manager::MEDIA,
					'default' => [
						'url' => esc_url('https://via.placeholder.com/1200x500/dddddd/dddddd/?text=slider'),
					],
				]
			);
			$repeater->add_control(
				'button_show_hide',
				[
					'label'        => __( 'Button', 'power-elements' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => __( 'show', 'power-elements' ),
					'label_off'    => __( 'Hide', 'power-elements' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$repeater->add_control(
				'link',
				[
					'label'       => __( 'Button Link', 'power-elements' ),
					'type'        => Controls_Manager::URL,
					'placeholder' => __( '#', 'power-elements' ),
					'default'     => [
						'url' => '#',
					],
					'condition'   => [
						'button_show_hide' => 'yes',
					],
				]
			);
			$repeater->add_control(
				'button_text',
				[
					'label'       => __( 'Button Text', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'Read More', 'power-elements' ),
					'placeholder' => __( 'Read More', 'power-elements' ),
					'condition'   => [
						'button_show_hide' => 'yes',
					]
				]
			);
			$repeater->add_control(
				'button_style',
				[
					'label'        => __( 'Type', 'power-elements' ),
					'type'         => Controls_Manager::SELECT,
					'default'      => 'power_slider_button effect_1',
					'options'      => [
						'power_slider_button effect_1'  => __( 'Effect 1', 'power-elements' ),
						'power_slider_button effect_2'  => __( 'Effect 2', 'power-elements' ),
						'power_slider_button effect_3'  => __( 'Effect 3', 'power-elements' ),
						'power_slider_button effect_4'  => __( 'Effect 4', 'power-elements' ),
						'power_slider_button effect_5'  => __( 'Effect 5', 'power-elements' ),
						'power_slider_button effect_6'  => __( 'Effect 6', 'power-elements' ),
						'power_slider_button effect_7'  => __( 'Effect 7', 'power-elements' ),
						'power_slider_button effect_8'  => __( 'Effect 8', 'power-elements' ),
						'power_slider_button effect_9'  => __( 'Effect 9', 'power-elements' ),
					]
				]
			);
			$repeater->add_control(
				'alignment',
				[
					'label'     => __( 'Alignment', 'power-elements' ),
					'type'      => Controls_Manager::CHOOSE,
					'default'	=> 'center',
					'options'   => [
						'left'    => [
							'title' => __( 'Left', 'power-elements' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power-elements' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power-elements' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power-elements' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors' => [
						'{{WRAPPER}} {{CURRENT_ITEM}} .slider_details' =>  'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'peeao_slider_list',
				[
					'label'       => __( 'Slider List', 'power-elements' ),
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'list_title'   => __( 'Title #1', 'power-elements' ),
							'list_content' => __( 'Item content.', 'power-elements' ),
							'peeao_slider_image' => [
								'url' => esc_url('https://via.placeholder.com/1200x500/dddddd/dddddd/?text=slider')
							]
						],
						[
							'list_title'   => __( 'Title #2', 'power-elements' ),
							'list_content' => __( 'Item content.', 'power-elements' ),
							'peeao_slider_image' => [
								'url' => esc_url('https://via.placeholder.com/1200x500/cccccc/cccccc/?text=slider')
							]
						],
					],
					'title_field' => '{{{ heading }}}',
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL SETTINGS
			 */
			$this->start_controls_section(
				'section_slider_carousel_settings',
				[
					'label' => __( 'Settings', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'peeao_slider_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power-elements' ),
					'type'        => Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_slider_carousel_item_mobile',
				[
					'label'       => __( 'Mobile Item', 'power-elements' ),
					'type'        => Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_slider_carousel_margin',
				[
					'label'   => __( 'Margin', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'peeao_slider_carousel_padding',
				[
					'label'   => __( 'Padding', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'peeao_slider_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_slider_carousel_responsive_class',
				[
					'label'   => __( 'Responsive Class', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_slider_carousel_dots',
				[
					'label'   => __( 'Dots', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_slider_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_slider_nav_style',
				[
					'label'   => __( 'Navigation Position', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'power_owl_nav_bottom',
					'options' => [
						'power_owl_nav_bottom'  => __( 'Bottom', 'power-elements' ),
						'power_owl_nav_side' => __( 'Side', 'power-elements' ),
					],
					'condition'   => [
						'peeao_slider_carousel_nav' => 'true',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE
			 */
			$this->start_controls_section(
				'peeao_slider_design',
				[
					'label' => __( 'Content Area', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_slider_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_slider_item',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'default'    => [
						'top' => '100',
						'right' => '50',
						'bottom' => '100',
						'left' => '50',
						'unit' => 'px',
						'isLinked' => '',
					],
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			

			/*
			 * Slider Tag Style
			 */
			$this->start_controls_section(
				'peeao_slider_sub_heading_style',
				[
					'label' => __( 'Sub Heading', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_slider_sub_heading_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slider_details .sub_heading',
				]
			);
			$this->add_control(
				'peeao_slider_sub_heading_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slider_details .sub_heading' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_slider_sub_heading_text_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .slider_details .sub_heading',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_sub_heading_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider_details .sub_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_sub_heading_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider_details .sub_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
			/*
			 * Heading Style
			 */
			$this->start_controls_section(
				'peeao_slider_heading_style',
				[
					'label' => __( 'Heading', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_slider_heading_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slider_details .heading',
				]
			);
			$this->add_control(
				'peeao_slider_heading_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slider_details .heading' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_slider_heading_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .slider_details .heading',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_heading_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider_details .heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_heading_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider_details .heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * Description Style
			 */
			$this->start_controls_section(
				'peeao_slider_description_style',
				[
					'label' => __( 'Description', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_slider_title_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slider_details .description',
				]
			);
			$this->add_control(
				'peeao_slider_description_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slider_details .description' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'peeao_slider_description_shadow',
					'label'    => __( 'Text Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .slider_details .description',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_description_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider_details .description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_description_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider_details .description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * STYLE BUTTON
			 */
			$this->start_controls_section(
				'peeao_slider_button_style',
				[
					'label' => __( 'Button', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'peeao_slider_button_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'peeao_slider_button_hover_color',
				[
					'label'     => __( 'Hover Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:hover' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_slider_button_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			$this->add_control(
				'peeao_slider_button_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:after' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'peeao_slider_button_bg_hover_color',
				[
					'label'     => __( 'Hover Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:before' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'peeao_slider_button_bg_hidden_color',
				[
					'label'     => __( 'Hidden Bg Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'peeao_slider_button_box_shadow',
					'label' => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_slider_button_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_button_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_button_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'default'    => [
						'top' => '30',
						'right' => '0',
						'bottom' => '0',
						'left' => '0',
						'unit' => 'px',
						'isLinked' => '',
					],
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button_group' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_button_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * STYLE NAVIGATION
			 */
			$this->start_controls_section(
				'peeao_slider_navigation_style',
				[
					'label' => __( 'Navigation', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);

			$this->start_controls_tabs(
				'peeao_slider_navigation_tab'
			);
			$this->start_controls_tab(
				'peeao_slider_navigation_normal_tab',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_slider_navigation_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next'=> 'color: {{VALUE}}',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_slider_navigation_background',
					'label'      => __( 'Background', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_slider_navigation_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_navigation_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_navigation_width',
				[
					'label'   => __( 'Width', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 2,
					],
					'default' => [
						'size' => 36,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_navigation_height',
				[
					'label'   => __( 'Height', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 2,
					],
					'default' => [
						'size' => 36,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next' => 'height: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_navigation_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_navigation_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_slider_navigation_box_shadow',
					'label'      => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next',
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'peeao_slider_navigation_hover_tab',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_slider_navigation_hover_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev:hover, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next:hover'=> 'color: {{VALUE}}',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_slider_navigation_hover_background',
					'label'      => __( 'Background', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev:hover, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next:hover',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_slider_navigation_hover_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev:hover, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next:hover',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_navigation_hover_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-prev:hover, {{WRAPPER}} .power_slider.owl-theme .owl-nav button.owl-next:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();

			$this->end_controls_tabs();
			$this->end_controls_section();


			/*
			 * STYLE DOT
			 */
			$this->start_controls_section(
				'peeao_slider_dot_style',
				[
					'label' => __( 'Dot', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);

			$this->start_controls_tabs(
				'peeao_slider_dot_tab'
			);
			$this->start_controls_tab(
				'peeao_slider_dot_normal_tab',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_slider_dot_background',
					'label'      => __( 'Background', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_slider_dot_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_dot_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_dot_width',
				[
					'label'   => __( 'Width', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 30,
						'step' => 2,
					],
					'default' => [
						'size' => 18,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_dot_height',
				[
					'label'   => __( 'Height', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 30,
						'step' => 2,
					],
					'default' => [
						'size' => 18,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot' => 'height: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_dot_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_slider_dot_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_slider_dot_box_shadow',
					'label'      => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot',
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'peeao_slider_dot_active_tab',
				[
					'label' => __( 'Active', 'power-elements' ),
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_slider_dot_active_background',
					'label'      => __( 'Background', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot.active',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_slider_dot_active_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector'   => '{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot.active',
				]
			);
			$this->add_responsive_control(
				'peeao_slider_dot_active_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider.owl-theme .owl-dots .owl-dot.active' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			
			$this->end_controls_tabs();
			$this->end_controls_section();
			
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings();
			$side_nav_class = ($settings['peeao_slider_nav_style'] == 'power_owl_nav_side') ? 'side_nav' : ''; 
			if ( $settings['peeao_slider_list'] ) {
				
				$html = '<div class="peeao-wrapper"><div class="owl-carousel owl-theme power_slider '.$side_nav_class.'" data-owl-carousel=\'{"items": ' . esc_attr( $settings['peeao_slider_carousel_item'] ) . ', "margin": ' . esc_attr( $settings['peeao_slider_carousel_margin']['size'] ) . ', "padding": ' . esc_attr( $settings['peeao_slider_carousel_padding']['size'] ) . ', "autoplay": ' . esc_attr( $settings['peeao_slider_carousel_autoplay'] ) . ', "responsiveClass": ' . esc_attr( $settings['peeao_slider_carousel_responsive_class'] ) . ', "dots": ' . esc_attr( $settings['peeao_slider_carousel_dots'] ) . ', "nav": ' . esc_attr( $settings['peeao_slider_carousel_nav'] ) . ',"responsive":{ "0" :{ "items": ' . esc_attr( $settings['peeao_slider_carousel_item_mobile'] ) . ' }, "540" :{ "items": ' . esc_attr( $settings['peeao_slider_carousel_item_mobile'] ) . ' }, "860" :{ "items" : ' . esc_attr( $settings['peeao_slider_carousel_item'] ) . ' } , "1000":{ "items" : ' . esc_attr( $settings['peeao_slider_carousel_item'] ) . ' }}}\'>';
				foreach ( $settings['peeao_slider_list'] as $item ) {
					if ( $item['button_show_hide']=='yes' ) {
					if ( ! empty( $item['link'] ) ) {
						$this->add_render_attribute( 'button', 'href', $item['link'] );
					}

					


					$button_html = '<div class="power_slider_button_group">
	                            <a ' . $this->get_render_attribute_string( 'button' ) . ' class="'.esc_html( $item['button_style'] ).'"><!--effect_(1-5)-->
	                                <span class="button_value">' . esc_html( $item['button_text'] ) . '</span>
	                            </a>
	                        </div>';
				} else {
					$button_html = '';
				}
					if ( $item['peeao_slider_image']['url'] != '' ) {
						$img = 'style="background-image:url(' . esc_url( $item['peeao_slider_image']['url'] ) . ')"';
					} else {
						$img = '';
					}
					
					if ( $item['description_text'] != '' ) {
						$description = esc_html( $item['description_text'] ) ;
					} else {
						$description = '';
					}
					
					
						$html .= ' <div class="item elementor-repeater-item-' . $item['_id'] . '">
					                <div class="power_slider_item" ' . $img . '>
										<div class="slider_inner">
											<div class="slider_details">
												<h3 class="sub_heading">' . esc_html( $item['sub_heading'] ) . '</h3>
												<div class="inner_heading">
													<h2 class="heading">' . esc_html( $item['heading'] ) . '</h2>
													<p class="description">' . $description . '</p>
												</div>
												' . $button_html . '
											</div>
										</div>
									</div>
				                </div>';
					
				}
				$html .= '</div></div>';
				
				echo $html;
			} else {
				echo esc_html__( 'Something wrong', 'power-elements' );
			}
			
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() { }
	}