<?php
namespace PowerElements;
use PowerElements\Helpers\Partials as Partials;
use PowerElements\Helpers\Utils as Utils;


use \Elementor\Repeater;
use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;

	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0 
	 */
	class PowerElements_Carousel extends Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_elements_carousel';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Carousel', 'power-elements' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-carousel';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'peeao_carousel_content_tab',
				[
					'label' => __( 'Content', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			
			$repeater = new Repeater();
			$repeater->add_control(
				'carousel_source',
				[
					'label'   => __( 'Source', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'custom',
					'options' => [
						'custom' => __( 'Custom', 'power-elements' ),
						'nested_template'   => __( 'Nested Templates', 'power-elements' )
					]
				]
			);
			$repeater->add_control(
				'nested_template_id',
				[
					'label'   => __( 'Nested Templates', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '',
					'options' => Partials::get_all_nested_templates(),
					'condition' => [
						'carousel_source' => 'nested_template',
					]
				]
			);
			$repeater->add_control(
				'carousel_content',
				[
					'label'       => __( 'Content', 'power-elements' ),
					'type'        => Controls_Manager::WYSIWYG,
					'default'     => __( 'You need no programming knowledge to edit this carousel. You will be able to edit change
					element there as you wish. So change according to the vibe of your website njoy your
					with us in the journey', 'power-elements' ),
					'placeholder' => __( 'Give your carousel content', 'power-elements' ),
					'condition' => [
						'carousel_source' => 'custom',
					]
				]
			);
			$this->add_control(
				'peeao_carousel_list',
				[
					'label'       => __( 'Carousel', 'power-elements' ),
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'carousel_content' => __( 'You need no programming knowledge to edit this carousel. You will be able to edit change
							element there as you wish. So change according to the vibe of your website njoy your
							with us in the journey', 'power-elements' ),
						],
						[
							'carousel_content' => __( 'You need no programming knowledge to edit this carousel. You will be able to edit change
							element there as you wish. So change according to the vibe of your website njoy your
							with us in the journey', 'power-elements' ),
						],
						[
							'carousel_content' => __( 'You need no programming knowledge to edit this carousel. You will be able to edit change
							element there as you wish. So change according to the vibe of your website njoy your
							with us in the journey', 'power-elements' ),
						],
					],
				]
			);
			$this->end_controls_section();

			/*
			 * CAROUSEL SETTINGS
			 */
			$this->start_controls_section(
				'peeao_carousel_section_settings_tab',
				[
					'label' => __( 'Settings', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'peeao_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power-elements' ),
					'type'        => Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 5,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_carousel_item_mobile',
				[
					'label'       => __( 'Mobile Item', 'power-elements' ),
					'type'        => Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 5,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_carousel_margin',
				[
					'label'   => __( 'Margin', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'peeao_carousel_padding',
				[
					'label'   => __( 'Padding', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'peeao_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_carousel_responsive_class',
				[
					'label'   => __( 'Responsive Class', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_carousel_dots',
				[
					'label'   => __( 'Dots', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power-elements' ),
						'false' => __( 'False', 'power-elements' ),
					],
				]
			);
			$this->add_control(
				'peeao_carousel_nav_style',
				[
					'label'   => __( 'Navigation Position', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'power_owl_nav_bottom',
					'options' => [
						'power_owl_nav_bottom'  => __( 'Bottom', 'power-elements' ),
						'power_owl_nav_side' => __( 'Side', 'power-elements' ),
					],
					'condition'   => [
						'peeao_carousel_nav' => 'true',
					],
				]
			);
			$this->end_controls_section();
			
			/*
			 * STYLE CONTENT WRAPPER
			 */
			$this->start_controls_section(
				'peeao_carousel_content_wrapper_style',
				[
					'label' => __( 'Carousel Wrapper', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'       => 'peeao_carousel_content_wrapper_background',
					'label'      => __( 'Background', 'power-elements' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .peeao-wrapper .power_carousel_container',
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_carousel_content_wrapper_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel_container',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_carousel_content_wrapper_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel_container',
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_content_wrapper_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel_container' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_content_wrapper_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel_container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_content_wrapper_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel_container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
			/*
			 * NAV Style
			 */
			$this->start_controls_section(
				'peeao_carousel_nav_style_tab',
				[
					'label'     => __( 'Nav', 'power-elements' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'peeao_carousel_nav' => 'true',
					],
				]
			);

			$this->add_responsive_control(
				'peeao_carousel_nav_align',
				[
					'label'     => __( 'Alignment', 'power-elements' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => __( 'Left', 'power-elements' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power-elements' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power-elements' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power-elements' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'default'  => 'center',
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav' => 'text-align: {{VALUE}};',
					],
					'condition'   => [
						'peeao_carousel_nav_style' => 'power_owl_nav_bottom',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_nav_size',
				[
					'label'   => __( 'Arrow Size', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px', 'em' ],
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 1,
					],
					'default' => [
						'size' => 30,
						'unit' => 'px',
					],
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-nav [class*=\'owl-\']' => 'font-size: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_nav_prev_spacing',
				[
					'label'   => __( 'Previous arrow spacing', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 1,
					],
					'default' => [
						'size' => 50,
					],
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev' => 'margin-left: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_nav_next_spacing',
				[
					'label'   => __( 'Next arrow spacing', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 1,
					],

					'default' => [
						'size' => 50,
					],
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next' => 'margin-right: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->start_controls_tabs(
				'peeao_carousel_nav_tabs'
			);
			
			$this->start_controls_tab(
				'peeao_carousel_nav_normal_tab',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_carousel_nav_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev' => 'color: {{VALUE}}',
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next' => 'color: {{VALUE}}',
						'{{WRAPPER}} .power_outside_grid_owl_prev, .power_outside_grid_owl_next' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_carousel_nav_bg_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev' => 'background-color: {{VALUE}}',
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next' => 'background-color: {{VALUE}}',
						'{{WRAPPER}} .power_outside_grid_owl_prev, .power_outside_grid_owl_next' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_carousel_nav_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next, {{WRAPPER}} .power_outside_grid_owl_next,{{WRAPPER}} .power_outside_grid_owl_prev',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_carousel_nav_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next, {{WRAPPER}} .power_outside_grid_owl_next,{{WRAPPER}} .power_outside_grid_owl_prev',
				]
			);
			$this->end_controls_tab();
			
			$this->start_controls_tab(
				'peeao_carousel_nav_hover_tab',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_carousel_nav_hover_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover ,{{WRAPPER}} .power_outside_grid_owl_prev:hover' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'peeao_carousel_nav_bg_hover_color',
				[
					'label'     => __( 'Background Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_carousel_nav_hover_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_carousel_nav_hover_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev:hover, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next:hover, {{WRAPPER}} .power_outside_grid_owl_next:hover,{{WRAPPER}} .power_outside_grid_owl_prev:hover',
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'peeao_carousel_nav_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'peeao_carousel_nav_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-prev, {{WRAPPER}} .peeao-wrapper .power_carousel  .owl-nav .owl-next' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * DOTS Style
			 */
			$this->start_controls_section(
				'peeao_carousel_dots_style_tab',
				[
					'label'     => __( 'Dots', 'power-elements' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'peeao_carousel_dots' => 'true',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_dots_spacing',
				[
					'label'   => __( 'Spacing', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
					'default' => [
						'size' => 5,
					],
					'selectors' => [
						'{{WRAPPER}} .power_carousel.owl-theme .owl-dots .owl-dot' => 'margin: 0 {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->start_controls_tabs(
				'peeao_carousel_dots_tabs'
			);
			
			$this->start_controls_tab(
				'peeao_carousel_dots_normal_tab',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_dots_height',
				[
					'label'   => __( 'Dot Height', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
					'default' => [
						'size' => 7,
					],
					'selectors' => [
						'{{WRAPPER}} .power_carousel.owl-theme .owl-dots .owl-dot span' => 'height: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_dots_width',
				[
					'label'   => __( 'Dot Width', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
					'default' => [
						'size' => 7,
					],
					'selectors' => [
						'{{WRAPPER}} .power_carousel.owl-theme .owl-dots .owl-dot span' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_control(
				'peeao_carousel_dots_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot span' => 'background: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_carousel_dots_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_carousel_dots_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot',
				]
			);
			$this->add_control(
				'peeao_carousel_dots_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'peeao_carousel_dots_active_tab',
				[
					'label' => __( 'Active', 'power-elements' ),
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_dots_height_active',
				[
					'label'   => __( 'Dot Height', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
					'default' => [
						'size' => 7,
					],
					'selectors' => [
						'{{WRAPPER}} .power_carousel.owl-theme .owl-dots .owl-dot.active span' => 'height: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_carousel_dots_width_active',
				[
					'label'   => __( 'Dot Width', 'power-elements' ),
					'type'    => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range'   => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
					'default' => [
						'size' => 7,
					],
					'selectors' => [
						'{{WRAPPER}} .power_carousel.owl-theme .owl-dots .owl-dot.active span' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_control(
				'peeao_carousel_dots_active_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot.active span' => 'background: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_carousel_dots_active_box_shadow',
					'label'    => __( 'Box Shadow', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot.active span',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'peeao_carousel_dots_active_border',
					'label'    => __( 'Border', 'power-elements' ),
					'selector' => '{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot.active span',
				]
			);
			$this->add_control(
				'peeao_carousel_dots_active_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .peeao-wrapper .power_carousel.owl-theme .owl-dots .owl-dot.active span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			
			$this->end_controls_tabs();
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings();
			$html     = '<div class="peeao-wrapper">';

			$side_nav_class = ($settings['peeao_carousel_nav_style'] == 'power_owl_nav_side') ? 'side_nav' : ''; 
			if ( $settings['peeao_carousel_list'] ) {
				
				$content_html = '<div class="power_carousel_container">';
				
				$content_html = '<div class="owl-carousel owl-theme power_carousel '.$side_nav_class.'" 
				data-owl-carousel=\'{"items": ' . esc_attr( $settings['peeao_carousel_item'] ) . ',
					"margin": ' . esc_attr( $settings['peeao_carousel_margin']['size'] ) . ',
					 "padding": ' . esc_attr( $settings['peeao_carousel_padding']['size'] ) . ', 
					 "autoplay": ' . esc_attr( $settings['peeao_carousel_autoplay'] ) . ', 
					 "responsiveClass": ' . esc_attr( $settings['peeao_carousel_responsive_class'] ) . ', 
					 "dots": ' . esc_attr( $settings['peeao_carousel_dots'] ) . ', 
					 "nav": ' . esc_attr( $settings['peeao_carousel_nav'] ) . ',
					 "responsive":{ "0" :{ 
						 "items": ' . esc_attr( $settings['peeao_carousel_item_mobile'] ) . ' }, 
						 "540" :{ "items": ' . esc_attr( $settings['peeao_carousel_item_mobile'] ) . ' }, 
						 "860" :{ "items" : ' . esc_attr( $settings['peeao_carousel_item'] ) . ' } , 
						 "1000":{ "items" : ' . esc_attr( $settings['peeao_carousel_item'] ) . ' }}}\'>';
				foreach ( $settings['peeao_carousel_list'] as $key => $item ) {
					$content_html .= '<div class="item">';
					if ( $key == 0 ) {
						$active = 'selected';
					} else {
						$active = 'not_selected'; 
					}

					if($item['carousel_source'] == 'custom'){
						$content_html .= $item['carousel_content'];
					}elseif($item['carousel_source'] == 'nested_template'){
						$content_html .= Utils::get_builder_content($item['nested_template_id']) ; 
					}
					$content_html .= '</div>';
				}
				$content_html .= '</div>';
				$content_html .= '</div>';

				$html         .= $content_html;
				$html         .= '</div>';
			}
			
			echo $html;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}