<?php

namespace PowerElements;

use PowerElements\Helpers\Utils as Utils;

use \Elementor\Widget_Base;
use \Elementor\Repeater;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Power Element social widget.
 *
 * Elementor widget that displays a social with the ability to control every
 * aspect of the social design.
 *
 * @since 1.0.0
 */
class PowerElements_Heading extends Widget_Base
{

	/**
	 * Get widget name.
	 *
	 * Retrieve social widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'power_elements_heading';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve social widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('Heading', 'power-elements');
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve social widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-t-letter';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the social widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['power-elements'];
	}

	/**
	 * Register social widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls()
	{
		$this->start_controls_section(
			'peeao_section_heading',
			[
				'label' => esc_html__('Heading Content', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_text',
			[
				'label'       => esc_html__('Title', 'power-elements'),
				'type'        => Controls_Manager::TEXT,
				'description'	=> esc_html__('Use this {focused text} format for focused parts of title.', 'power-elements'),
				'default'     => esc_html__('Heading', 'power-elements'),
				'placeholder' => esc_html__('Title', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_text_enable_typed',
			[
				'label' => esc_html__('Enable Typed? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);
		$repeater = new Repeater();
		$repeater->add_control(
			'typed',
			[
				'label' => esc_html__('Item', 'power-elements'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
			]
		);
		$this->add_control(
			'peeao_heading_typed_items',
			[
				'label' => esc_html__('Typed Items', 'power-elements'),
				'type' => Controls_Manager::REPEATER,
				'title_field' => '{{ typed }}',
				'default' => [
					['typed' => esc_html__('Charity', 'power-elements')],
					['typed' => esc_html__('Educational', 'power-elements')],
					['typed' => esc_html__('Health', 'power-elements')],
				],
				'condition' => [
					'peeao_heading_text_enable_typed' => 'yes'
				],
				'fields' => $repeater->get_controls(),
			]
		);
		$this->add_control(
			'peeao_heading_subtitle',
			[
				'label'       => esc_html__('Sub Title', 'power-elements'),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__('Subtitle', 'power-elements'),
				'placeholder' => esc_html__('Subtitle', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_text_para',
			[
				'label'       => esc_html__('Praragraph', 'power-elements'),
				'type'        => Controls_Manager::WYSIWYG,
				'default'     => esc_html__('All the elements in this element are customizable.
					You need no programming Knowledge to edit if you want
					to edit this section. ', 'power-elements'),
				'placeholder' => esc_html__('Description', 'power-elements'),
			]
		);

		$this->end_controls_section();
		$this->start_controls_section(
			'peeao_section_heading_settings',
			[
				'label' => esc_html__('Settings', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_style',
			[
				'label'   => esc_html__('Type', 'power-elements'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'stp',
				'options' => [
					'stp' => esc_html__('S-T-P', 'power-elements'),
					'tsp' => esc_html__('T-S-P', 'power-elements')
				]
			]
		);

		$this->add_control(
			'peeao_heading_subtitle_before_after__switch',
			[
				'label' => esc_html__('Show Subtitle Border? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);

		$this->add_control(
			'peeao_heading_divder_before_show',
			[
				'label' => esc_html__('Show Top Divider? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_divder_after_show',
			[
				'label' => esc_html__('Show Bottom Divider? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'nos',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);

		$this->add_responsive_control(
			'peeao_heading_align',
			[
				'label'     => esc_html__('Alignment', 'power-elements'),
				'type'      => Controls_Manager::CHOOSE,
				'options' => [
					'left'    => [
						'title' => esc_html__('Left', 'power-elements'),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'power-elements'),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => esc_html__('Right', 'power-elements'),
						'icon' => 'fa fa-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section(); 

		$this->start_controls_section(
			'peeao_heading_section_style',
			[
				'label' => esc_html__('Title', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'peeao_heading_title_bg_use',
			[
				'label' => esc_html__('Use BG? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_title_bg_inverse',
			[
				'label' => esc_html__('BG inverse? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
				'condition'		=> [
					'peeao_heading_title_bg_use' => 'yes'
				]
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'peeao_heading_title_bg',
				'label'		 => esc_html__('BG', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title',
				'condition' => [
					'peeao_heading_title_bg_use' => 'yes',
				],
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'peeao_heading_title_typography',
				'label'    => esc_html__('Typography', 'power-elements'),
				'fields_options' => [
					'text_transform' => [
						'default' => 'normal'
					]
				],
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title',
			]
		);
		$this->add_control(
			'peeao_heading_title_text_color',
			[
				'label'     => esc_html__('Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_title' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_title_margin',
			[
				'label'      => esc_html__('Margin', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_title_padding',
			[
				'label'      => esc_html__('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_title_text_shadow',
				'label'    => esc_html__('Text Shadow', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'peeao_heading_focused_title_section_style',
			[
				'label' => esc_html__('Focused Title', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'peeao_heading_focused_title_bg_use',
			[
				'label' => esc_html__('Use BG? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_focused_title_bg_inverse',
			[
				'label' => esc_html__('BG inverse? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
				'condition'		=> [
					'peeao_heading_focused_title_bg_use' => 'yes'
				]
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'peeao_heading_focused_title_bg',
				'label'		 => esc_html__('Background', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title span',
				'condition' => [
					'peeao_heading_focused_title_bg_use' => 'yes',
				],
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'peeao_heading_focused_title_typography',
				'label'    => esc_html__('Typography', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title span',
			]
		);
		$this->add_control(
			'peeao_heading_focused_title_text_color',
			[
				'label'     => esc_html__('Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_title span' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_focused_title_margin',
			[
				'label'      => esc_html__('Margin', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_title span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_focused_title_padding',
			[
				'label'      => esc_html__('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_title span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_focused_title_text_shadow',
				'label'    => esc_html__('Text Shadow', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title span',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'peeao_heading_typed_title_section_style',
			[
				'label' => esc_html__('Typed Title', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'peeao_heading_typed_title_bg_use',
			[
				'label' => esc_html__('Use BG? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_typed_title_bg_inverse',
			[
				'label' => esc_html__('BG inverse? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
				'condition'		=> [
					'peeao_heading_typed_title_bg_use' => 'yes'
				]
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'peeao_heading_typed_title_bg',
				'label'		 => esc_html__('Background', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title span.power_typed',
				'condition' => [
					'peeao_heading_typed_title_bg_use' => 'yes',
				],
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'peeao_heading_typed_title_typography',
				'label'    => esc_html__('Typography', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title span.power_typed',
			]
		);
		$this->add_control(
			'peeao_heading_typed_title_text_color',
			[
				'label'     => esc_html__('Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_title span.power_typed' => 'color: {{VALUE}};',
					'{{WRAPPER}} .power_heading_wrapper .power_title .typed-cursor' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_typed_title_margin',
			[
				'label'      => esc_html__('Margin', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_title span.power_typed' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_typed_title_padding',
			[
				'label'      => esc_html__('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_title span.power_typed' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_typed_title_text_shadow',
				'label'    => esc_html__('Text Shadow', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title span.power_typed',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'peeao_heading_subtitle_style',
			[
				'label' => esc_html__('Sub Title', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'peeao_heading_subtitle_bg_use',
			[
				'label' => esc_html__('Use BG? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
			]
		);
		$this->add_control(
			'peeao_heading_subtitle_bg_inverse',
			[
				'label' => esc_html__('BG inverse? ', 'power-elements'),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => esc_html__('Yes', 'power-elements'),
				'label_off' => esc_html__('No', 'power-elements'),
				'condition'		=> [
					'peeao_heading_subtitle_bg_use' => 'yes'
				]
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'peeao_heading_subtitle_bg',
				'label'		 => esc_html__('BG', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
				'condition' => [
					'peeao_heading_subtitle_bg_use' => 'yes',
				],
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'peeao_heading_sub_typography',
				'label'    => esc_html__('Typography', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
			]
		);
		$this->add_control(
			'peeao_heading_sub_text_color',
			[
				'label'     => esc_html__('Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'peeao_heading_sub_seperator_color',
			[
				'label'     => esc_html__('Seperator Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::after' => 'background: {{VALUE}};',
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::before' => 'background: {{VALUE}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_sub_margin',
			[
				'label'      => esc_html__('Margin', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_sub_padding',
			[
				'label'      => esc_html__('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'peeao_heading_sub_title_bg',
				'label'    => __('Background', 'power-elements'),
				'types'    => ['classic', 'gradient', 'video'],
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'peeao_heading_sub_title_border',
				'label'    => __('Border', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
			]
		);
		$this->add_responsive_control(
			'peeao_heading_sub_title_radius',
			[
				'label'      => __('Border Radius', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_sub_title_box_shadow',
				'selector' => '{{WRAPPER}} .power_iconbox:hover',
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_sub_title_text_shadow',
				'label'    => esc_html__('Text Shadow', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'peeao_heading_description_style',
			[
				'label' => esc_html__('Description', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'peeao_heading_para_typography',
				'label'    => esc_html__('Paragraph Typography', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_shortdesc',
			]
		);
		$this->add_control(
			'peeao_heading_para_text_color',
			[
				'label'     => esc_html__('Paragraph Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_shortdesc' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_para_margin',
			[
				'label'      => esc_html__('Paragraph Margin', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_shortdesc' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_para_padding',
			[
				'label'      => esc_html__('Paragraph Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper .power_shortdesc' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_para_text_shadow',
				'label'    => esc_html__('Paragraph Text Shadow', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_heading_wrapper .power_shortdesc',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'peeao_heading_others_style',
			[
				'label' => esc_html__('Others', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'peeao_heading_subtitle_before_after_width',
			[
				'label' => esc_html__('Subtitle border Width', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 85,
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::before' => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::after' => 'width: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'peeao_heading_subtitle_before_after__switch' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_heading_subtitle_before_after_height',
			[
				'label' => esc_html__('Subtitle border height', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 1,
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::before' => 'height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::after' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'peeao_heading_subtitle_before_after__switch' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_heading_subtitle_before_after_spacing',
			[
				'label' => esc_html__('Subtitle border spacing', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 50,
						'max' => 200,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 70,
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::before' => 'right: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .power_heading_wrapper .power_subtitle.power-heading-subtitle-with-border::after' => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'peeao_heading_subtitle_before_after__switch' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_heading_before_bg_color',
			[
				'label'     => esc_html__('Top Divider Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_top_divider' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'peeao_heading_divder_before_show' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_heading_before_width',
			[
				'label' => esc_html__('Top Divider Width', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_top_divider' => 'width: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'peeao_heading_divder_before_show' => 'yes',
				],
			]
		);

		$this->add_control(
			'peeao_heading_before_height',
			[
				'label' => esc_html__('Top Divider Height', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'condition' => [
					'peeao_heading_divder_before_show' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_top_divider' => 'height: {{SIZE}}{{UNIT}};',
				]
			]
		);
		$this->add_control(
			'peeao_heading_before_spacing',
			[
				'label' => esc_html__('Top Divider Spacing', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'condition' => [
					'peeao_heading_divder_before_show' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_top_divider' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'peeao_heading_after_bg_color',
			[
				'label'     => esc_html__('Bottom Divider Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_bottom_divider' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'peeao_heading_divder_after_show' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_heading_after_width',
			[
				'label' => esc_html__('Bottom Divider Width', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'condition' => [
					'peeao_heading_divder_after_show' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_bottom_divider' => 'width: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'peeao_heading_after_height',
			[
				'label' => esc_html__('Bottom Divider Height', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'condition' => [
					'peeao_heading_divder_after_show' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_bottom_divider' => 'height: {{SIZE}}{{UNIT}};',
				]
			]
		);
		$this->add_control(
			'peeao_heading_after_spacing',
			[
				'label' => esc_html__('Bottom Divider Spacing', 'power-elements'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'condition' => [
					'peeao_heading_divder_after_show' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .power_heading_wrapper .power_heading_bottom_divider' => 'margin-top: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'peeao_heading_box_shadow',
				'selector' => '{{WRAPPER}} .power_heading_wrapper',
			]
		);
		$this->add_responsive_control(
			'peeao_heading_margin',
			[
				'label'      => esc_html__('Heading Margin', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_heading_padding',
			[
				'label'      => esc_html__('Heading Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .power_heading_wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
	}


	/**
	 * Render social widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render(){

		$settings = $this->get_settings();
		extract($settings);

		$title_bg_inverse = ($peeao_heading_title_bg_use == 'yes' && $peeao_heading_title_bg_inverse == 'yes') ? 'power-bg-inverse' : '';
		$sub_title_bg_inverse = ($peeao_heading_subtitle_bg_use == 'yes' && $peeao_heading_subtitle_bg_inverse == 'yes') ? 'power_focused_sub power-bg-inverse' : 'power_focused_sub';
		$focused_title_bg_inverse = ($peeao_heading_focused_title_bg_use == 'yes' && $peeao_heading_focused_title_bg_inverse == 'yes') ? 'power_focused power-bg-inverse' : 'power_focused';
		$typed_title_bg_inverse = ($peeao_heading_typed_title_bg_use == 'yes' && $peeao_heading_typed_title_bg_inverse == 'yes') ? 'power-bg-inverse' : '';
		$compiled_focused_title = str_replace(['{', '}'], ['<span class="' . esc_attr($focused_title_bg_inverse) . ' ">', '</span>'], Utils::kses($peeao_heading_text));
		if ($peeao_heading_text_enable_typed == 'yes' && $peeao_heading_typed_items != '') {
			$typed_items = '<div style="display:none;" class="power-typed-strings power-typed-strings-'. $this->get_id().'">';
			foreach ($peeao_heading_typed_items as $item) {
				$typed_items .= '<p style="color="red">' . $item['typed'] . '</p>';
			}
			$typed_items .= '</div>';
			$compiled_title = str_replace(['[', ']'], ['<span class="power_typed power_typed_'. $this->get_id().' ' . $typed_title_bg_inverse . '">', '</span>'], Utils::kses($compiled_focused_title));
		
		} else {
			$typed_items = '';
			$compiled_title = $compiled_focused_title;
		}

		$peeao_heading_subtitle_before_after__switch = ($peeao_heading_subtitle_before_after__switch == 'yes') ? 'power-heading-subtitle-with-border' : '';

		$title    = ($peeao_heading_text != '') ?
			'<h2 class="power_title ' . esc_attr($title_bg_inverse) . '">
					' . Utils::kses($compiled_title) . '
				</h2>' . Utils::kses($typed_items) . '' : '';

		$subtitle = ($peeao_heading_subtitle != '') ? '<h5 class="power_subtitle ' . $peeao_heading_subtitle_before_after__switch . ' ' . esc_attr($sub_title_bg_inverse) . '">' . Utils::kses($peeao_heading_subtitle) . '</h5>' : '';
		$para     = (trim($peeao_heading_text_para) != '') ? '<p class="power_shortdesc">' . Utils::kses($peeao_heading_text_para) . '</p>' : '';
		$top_divider = ($peeao_heading_divder_before_show == 'yes') ? '<span class="power_heading_top_divider "></span><span class="power_heading_sub_divider "></span>' : '';
		$bottom_divider = ($peeao_heading_divder_after_show == 'yes') ? '<span class="power_heading_bottom_divider "></span>' : '';
		$html = ($peeao_heading_style == 'stp') ?
			$top_divider . $subtitle . $title . $para . $bottom_divider :
			$top_divider . $title . $subtitle . $para . $bottom_divider;

		echo '<div class="peeao-wrapper">';
		echo '<div class="power_heading_wrapper ' . $peeao_heading_align . '">';
		echo Utils::kses($html);
		echo '</div>';
		echo '</div>';
	}

	/**
	 * Render social widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template()
	{
	}
}
