<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Video extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_video';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Video', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-slider-video';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-video' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_image_carousel_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'video_type',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'youtube',
					'options' => [
						'youtube'     => __( 'YouTube', 'power_elements_elementor' ),
						'vimeo'       => __( 'Vimeo', 'power_elements_elementor' ),
						'dailymotion' => __( 'Daily Motion', 'power_elements_elementor' ),
						'self'        => __( 'Self Hosted', 'power_elements_elementor' )
					],
				]
			);
			$this->add_control(
				'video_url',
				[
					'label'       => __( 'Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true
					],
					'placeholder' => __( 'Enter your URL', 'power_elements_elementor' ),
					'default'     => '',
					'label_block' => true,
					'condition'   => [
						'video_type!' => 'self',
					],
				]
			);
			$this->add_control(
				'external_url',
				[
					'label'     => __( 'External URL', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type' => 'self',
					],
				]
			);
			$this->add_control(
				'self_url',
				[
					'label'       => __( 'Self Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true
					],
					'placeholder' => __( 'Enter your URL', 'power_elements_elementor' ),
					'default'     => '',
					'label_block' => true,
					'condition'   => [
						'external_url' => 'yes',
					],
				]
			);
			$this->add_control(
				'hosted_url',
				[
					'label'      => __( 'Choose MP4 File', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::MEDIA,
					'dynamic'    => [
						'active' => true,
					],
					'media_type' => 'video',
					'condition'  => [
						'external_url' => '',
						'video_type'   => 'self',
					],
				]
			);
			$this->add_control(
				'width',
				[
					'label'      => __( 'Height', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 1000,
							'step' => 5,
						],
						'%'  => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 400,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_video_content iframe' => 'height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'aspect_ratio',
				[
					'label'   => __( 'Aspect Ratio', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'power_video_aspect_ratio_3_2',
					'options' => [
						'power_video_aspect_ratio_3_2'  => __( '3:2', 'power_elements_elementor' ),
						'power_video_aspect_ratio_16_9' => __( '16:9', 'power_elements_elementor' ),
						'power_video_aspect_ratio_4_3'  => __( '4:3', 'power_elements_elementor' ),
						'power_video_aspect_ratio_1_1'  => __( '1:1', 'power_elements_elementor' )
					],
				]
			);
			$this->add_control(
				'more_options',
				[
					'label'     => __( 'Additional Options', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
					'condition' => [
						'video_type!' => 'self',
					],
				]
			);
			$this->add_control(
				'autoplay',
				[
					'label'     => __( 'Autoplay', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type!' => array('self','dailymotion'),
					],
				]
			);
			$this->add_control(
				'mute',
				[
					'label'     => __( 'Mute', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type!' => 'self',
					],
				]
			);
			$this->add_control(
				'loop',
				[
					'label'     => __( 'Loop', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type!' => 'self',
					],
				]
			);
			$this->add_control(
				'controls',
				[
					'label'     => __( 'Control', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type!' => array('self','vimeo'),
					],
				]
			);
			$this->add_control(
				'title',
				[
					'label'     => __( 'Title', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type!' => 'self',
						'video_type'  => 'vimeo',
					],
				]
			);
			$this->add_control(
				'portrait',
				[
					'label'     => __( 'Portrait', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type!' => 'self',
						'video_type'  => 'vimeo',
					],
				]
			);
			$this->add_control(
				'start_info',
				[
					'label'     => __( 'Start Info', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'condition' => [
						'video_type'  => 'dailymotion',
					],
				]
			);
			$this->add_control(
				'color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'condition' => [
						'video_type!' => array('self','youtube'),
					],
				]
			);
			$this->add_control(
				'start_time',
				[
					'label'       => __( 'Start Time', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 0,
					'max'         => 1000,
					'step'        => 1,
					'default'     => '',
					'description' => __( 'Start time in seconds', 'power_elements_elementor' ),
					'condition'   => [
						'video_type' => 'youtube',
					],
				]
			);
			$this->add_control(
				'end_time',
				[
					'label'       => __( 'End Time', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 0,
					'max'         => 1000000,
					'step'        => 1,
					'default'     => '',
					'description' => __( 'End time in seconds', 'power_elements_elementor' ),
					'condition'   => [
						'video_type' => 'youtube',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$video_url = $settings['video_url'];
			if ( 'self' === $settings['video_type'] ) {
				$video_url = $settings['self_url'];
			}
			
			if ( empty( $video_url ) ) {
				return;
			}
			if ( 'dailymotion' === $settings['video_type'] ) {
				$url = parse_url($video_url);
				$url= $url['path'];
				$parts = explode('/',$url);
				//$parts = explode('_',$parts[2]);
				if (in_array('embed',$parts)) {
				    $video_url = 'https://www.dailymotion.com/video/'.$parts[3];
				}
            }
			if ( 'self' === $settings['video_type'] ) {
				echo ' <div class="power_video_content ' . esc_attr( $settings['aspect_ratio'] ) . '">
                <video controls autoplay class="power_video_iframe">
                    <source src="' . esc_url( $video_url ) . '" type="video/mp4" />
                </video>
            </div>';
			} else {
				$embed_params = $this->get_embed_params();
				$video_html   = \Elementor\Embed::get_embed_html( $video_url, $embed_params );
				echo '<div class="power_video_content ' . esc_attr( $settings['aspect_ratio'] ) . '">';
				echo $video_html;
				echo '</div>';
			}
		}
		
		/**
		 * Get embed params.
		 *
		 * Retrieve video widget embed parameters.
		 *
		 * @since 1.5.0
		 * @access public
		 *
		 * @return array Video embed parameters.
		 */
		public function get_embed_params() {
			$settings = $this->get_settings_for_display();
			
			$params = [];
			
			if ( $settings['autoplay'] ) {
				$params['autoplay'] = '1';
			}
			
			$params_dictionary = [];
			
			if ( 'youtube' === $settings['video_type'] ) {
				$params_dictionary = [
					'loop',
					'controls',
					'mute',
				];
				
				if ( $settings['loop'] ) {
					$video_properties = \Elementor\Embed::get_video_properties( $settings['video_url'] );
					
					$params['playlist'] = $video_properties['video_id'];
				}
				
				$params['start'] = $settings['start_time'];
				
				$params['end'] = $settings['end_time'];
				
				$params['wmode'] = 'opaque';
			} elseif ( 'vimeo' === $settings['video_type'] ) {
				$params_dictionary = [
					'loop',
					'mute'     => 'muted',
					'title'    => 'title',
					'portrait' => 'portrait'
				];
				
				$params['color']     = str_replace( '#', '', $settings['color'] );
				$params['autopause'] = '0';
			} elseif ( 'dailymotion' === $settings['video_type'] ) {
				$params_dictionary = [
					'controls',
					'mute',
					'start_info' => 'ui-start-screen-info',
				];
				
				$params['ui-highlight'] = str_replace( '#', '', $settings['color'] );
				
				$params['endscreen-enable'] = '0';
			}
			
			foreach ( $params_dictionary as $key => $param_name ) {
				$setting_name = $param_name;
				
				if ( is_string( $key ) ) {
					$setting_name = $key;
				}
				
				$setting_value = $settings[ $setting_name ] ? '1' : '0';
				
				$params[ $param_name ] = $setting_value;
			}
			
			return $params;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#
            var url = settings.video_url;
            
            var autoplay = settings.autoplay ? 1:0;
            var mute = settings.mute ? 1:0;
            var loop = settings.loop ? 1:0;
            var loop = settings.loop ? 1:0;
            var controls = settings.controls ? 1:0;
            var title = settings.title ? 1:0;
            var portrait = settings.portrait ? 1:0;
            var start_info = settings.start_info ? 1:0;
            var color = settings.color ? settings.color.replace("#", ""):'';
            var start_time = settings.start_time;
            var end_time = settings.end_time;
            
            if(settings.video_type=='youtube'){
                var regExp = /^.*(youtu.be\/|v\/|u\/\w\/|embed\/|watch\?v=|\&v=)([^#\&\?]*).*/;
                var match = url.match(regExp);
                if (match && match[2].length == 11) {
                 url =  match[2];
                }
                url = 'https://www.youtube.com/embed/'+url+'?start='+start_time+'&end='+end_time+'&autoplay='+autoplay+'&mute='+mute+'&loop='+loop+'&controls='+controls;
            #>
            <div class="power_video_content {{settings.aspect_ratio}}">
                <iframe src="{{url}}"></iframe>
            </div>
            <#}else if(settings.video_type=='vimeo'){
                url =url+'?color='+color+'&autoplay='+autoplay+'&loop='+loop+'&muted='+mute+'&title='+title+'&portrait='+portrait;
            #>
            <div class="power_video_content {{settings.aspect_ratio}}">
                <iframe src="{{url}}"></iframe>
            </div>
            <#}else if(settings.video_type=='dailymotion'){
                var m = url.match(/^.+dailymotion.com\/(video|hub)\/([^_]+)[^#]*(#video=([^_&]+))?/);
                if (m !== null) {
                    if(m[4] !== undefined) {
                    url = m[4];
                    }
                    url = m[2];
                    url = 'https://www.dailymotion.com/embed/video/'+url+'?ui-highlight='+color+'&controls='+controls+'&ui-start-screen-info='+start_info+'&mute='+mute;
                }else{
                    url = url+'?ui-highlight='+color+'&controls='+controls+'&ui-start-screen-info='+start_info+'&mute='+mute;
                } #>
            <div class="power_video_content {{settings.aspect_ratio}}">
                <iframe src="{{url}}"></iframe>
            </div>
            <#}else if(settings.video_type=='self'){
                url = url; #>
            <div class="power_video_content {{settings.aspect_ratio}}">
                <video controls autoplay class="power_video_iframe">
                    <source src="{{url}}" type="video/mp4" />
                </video>
            </div>
            <#}
            #>
			<?php
		}
	}