<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Tabs extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_tabs';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Tab', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-tabs';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-tab' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-tab-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_tab_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'tab_style',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'vertical_style',
					'options' => [
						'horizontal_style' => __( 'Horizontal', 'power_elements_elementor' ),
						'vertical_style'   => __( 'Vertical', 'power_elements_elementor' )
					]
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'tab_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Title', 'power_elements_elementor' ),
					'placeholder' => __( 'Give your first tab title', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'tab_content',
				[
					'label'       => __( 'Content', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::WYSIWYG,
					'default'     => __( 'Content', 'power_elements_elementor' ),
					'placeholder' => __( 'Give your tab content', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'list',
				[
					'label'       => __( 'Tabs', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'list_title'   => __( 'Title #1', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
						[
							'list_title'   => __( 'Title #2', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
					],
					'title_field' => '{{{ tab_title }}}'
				]
			);
			$this->add_control(
				'tab_icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
					'placeholder' => __( 'Leave empty to hide', 'power_elements_elementor' ),
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE TAB
			 */
			$this->start_controls_section(
				'power_element_tab_style',
				[
					'label' => __( 'Tab', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'tab_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE TITLE WRAPPER
			 */
			$this->start_controls_section(
				'power_element_tab_title_wrapper_style',
				[
					'label' => __( 'Title Wrapper', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'title_wrapper_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'title_wrapper_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'title_wrapper_width',
				[
					'label'      => __( 'Width', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 1000,
							'step' => 5,
						],
						'%'  => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => '%',
						'size' => 25,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
			/*
			 * STYLE CONTENT WRAPPER
			 */
			$this->start_controls_section(
				'power_element_tab_content_wrapper_style',
				[
					'label' => __( 'Content Wrapper', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'content_wrapper_width',
				[
					'label'      => __( 'Width', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 1000,
							'step' => 5,
						],
						'%'  => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => '%',
						'size' => 25,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_container' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
			/*
			 * STYLE TITLE
			 */
			$this->start_controls_section(
				'power_element_tab_title_style',
				[
					'label' => __( 'Title', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'style_tab_title_tabs'
			);
			/*
			 * TITLE ACTIVE
			 */
			$this->start_controls_tab(
				'style_tab_title_active_tab',
				[
					'label' => __( 'Active', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'tab_title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected',
				]
			);
			$this->add_control(
				'tab_title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_title_bg_color',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'tab_title_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'tab_title_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected',
				]
			);
			$this->add_control(
				'tab_title_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a.selected' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * TITLE NORMAL
			 */
			$this->start_controls_tab(
				'style_tab_title_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'tab_title_typography_normal',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)',
				]
			);
			$this->add_control(
				'tab_title_color_normal',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_title_bg_color_normal',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'tab_title_shadow_normal',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'tab_title_border_normal',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)',
				]
			);
			$this->add_control(
				'tab_title_radius_normal',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_title_margin_normal',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_title_padding_normal',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:not(.selected)' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * TITLE HOVER
			 */
			$this->start_controls_tab(
				'style_tab_title_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'tab_title_typography_hover',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover',
				]
			);
			$this->add_control(
				'tab_title_color_hover',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_title_bg_color_hover',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'tab_title_shadow_hover',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'tab_title_border_hover',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover',
				]
			);
			$this->add_control(
				'tab_title_radius_hover',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_title_margin_hover',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_title_padding_hover',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button_group li a:hover' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * ICON STYLE
			 */
			$this->start_controls_section(
				'power_element_tab_icon_style',
				[
					'label' => __( 'Icon', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'style_tab_icon_tabs'
			);
			/*
			 * ICON ACTIVE
			 */
			$this->start_controls_tab(
				'style_tab_icon_active_tab',
				[
					'label' => __( 'Active', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'tab_icon_color_active',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a.selected .power_tab_button_icon i' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_icon_font_size_active',
				[
					'label' => __( 'Font Size', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 50,
							'step' => 1,
						],
						'em' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 18,
					],
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a.selected .power_tab_button_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * ICON NORMAL
			 */
			$this->start_controls_tab(
				'style_tab_icon_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'tab_icon_color_normal',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a:not(.selected) .power_tab_button_icon i' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_icon_font_size_normal',
				[
					'label' => __( 'Font Size', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 50,
							'step' => 1,
						],
						'em' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 18,
					],
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a:not(.selected) .power_tab_button_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * ICON HOVER
			 */
			$this->start_controls_tab(
				'style_tab_icon_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'tab_icon_color_hover',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a:hover .power_tab_button_icon i' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_icon_font_size_hover',
				[
					'label' => __( 'Font Size', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 50,
							'step' => 1,
						],
						'em' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 18,
					],
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a:hover .power_tab_button_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'hr',
				[
					'type' => \Elementor\Controls_Manager::DIVIDER,
				]
			);
			$this->add_control(
				'tab_icon_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a .power_tab_button_icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_icon_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_button a .power_tab_button_icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * CONTENT STYLE
			 */
			$this->start_controls_section(
				'power_element_tab_content_style',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'tab_content_typo',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info',
				]
			);
			$this->add_control(
				'tab_content_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'tab_content_bg_color',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'tab_content_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'tab_content_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info',
				]
			);
			$this->add_control(
				'tab_content_border_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_content_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'tab_content_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_tab_nav .power_tab_container .power_tab_info' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			$html     = '';
			if ( $settings['tab_icon'] != '' ) {
				$icon_html = '<span class="power_tab_button_icon"><i class="' . esc_attr( $settings['tab_icon'] ) . '"></i></span>';
			} else {
				$icon_html = '';
			}
			if ( $settings['list'] ) {
				$html         = ' <div class="power_tab_nav power_tab ' . esc_attr( $settings['tab_style'] ) . '">';
				$title_html   = '<ul class="power_tab_button_group">';
				$content_html = '<div  class="power_tab_container">';
				foreach ( $settings['list'] as $key => $item ) {
					if ( $key == 0 ) {
						$active = 'selected';
					} else {
						$active = 'not_selected';
					}
					$title_html   .= '<li class="power_tab_button ' . esc_attr( $active ) . '">
                        <a class="' . esc_attr( $active ) . '" href="#">
                            ' . $icon_html . esc_html( $item['tab_title'] ) . '
                        </a>
                    </li>';
					$content_html .= '<div class="power_tab_info ' . esc_attr( $active ) . '">
                        ' . $item['tab_content'] . '
                    </div>';
				}
				$title_html   .= '</ul>';
				$content_html .= '</div>';
				$html         .= $title_html . $content_html . '</div>';
			}
			
			echo $html;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <div class="power_tab_nav power_tab {{settings.tab_style}}">

                <ul class="power_tab_button_group">
                    <#
                    var active='';
                    _.each( settings.list, function( item , i) {
                    if(i==0){
                    active = 'selected';
                    }else{
                    active ='';
                    }
                    #>
                    <li class="power_tab_button {{active}}">
                        <a class="{{active}" href="#">
                            <# if(settings.tab_icon!=''){ #>
                            <span class="power_tab_button_icon"><i class="{{settings.tab_icon}}"></i></span>
                            <# } #>
                            {{{item.tab_title}}}
                        </a>
                    </li>
                    <# }); #>
                </ul>

                <div class="power_tab_container">
                    <#
                    var active='';
                    _.each( settings.list, function( item , i) {
                    if(i==0){
                    active = 'selected';
                    }else{
                    active ='';
                    }
                    #>
                    <div class="power_tab_info {{active}}">
                        {{{item.tab_content}}}
                    </div>
                    <# }); #>
                </div>
            </div>
			<?php
		}
	}