<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Portfolio_Carousel extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_portfolio_carousel';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Portfolio Carousel', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-posts-carousel';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-portfolio-carousel' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-portfoliocarousel-js' ];
		}

		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_posts_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_group_control(
				'powerpost',
				[
					'name' => 'powerpost',
				]
			);
			$this->add_control(
				'advanced',
				[
					'label' => __( 'Advanced', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING
				]
			);
			$this->add_control(
				'orderby',
				[
					'label' => __( 'Order By', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'default' => 'post_date',
					'options' => [
						'post_date' => __( 'Date', 'power_elements_elementor' ),
						'post_title' => __( 'Title', 'power_elements_elementor' ),
						'menu_order' => __( 'Menu Order', 'power_elements_elementor' ),
						'rand' => __( 'Random', 'power_elements_elementor' ),
					]
				]
			);
			
			$this->add_control(
				'order',
				[
					'label' => __( 'Order', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'default' => 'desc',
					'options' => [
						'asc' => __( 'ASC', 'power_elements_elementor' ),
						'desc' => __( 'DESC', 'power_elements_elementor' ),
					]
				]
			);
			
			$this->add_control(
				'offset',
				[
					'label' => __( 'Offset', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::NUMBER,
					'default' => 0,
					'description' => __( 'Use this setting to skip over posts (e.g. \'2\' to skip over 2 posts).', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'post__not_in',
				[
					'label' => __('Exclude', 'power_elements_elementor'),
					'type' => \Elementor\Controls_Manager::SELECT2,
					'options' => Power_elements_Helper::power_get_all_types_post(),
					'label_block' => true,
					'post_type' => '',
					'multiple' => true,
					'condition' => [
						'eaeposts_post_type!' => 'by_id',
					],
				]
			);
			$this->add_control(
				'posts_per_page',
				[
					'label' => __('Posts Per Page', 'power_elements_elementor'),
					'type' => \Elementor\Controls_Manager::NUMBER,
					'default' => '4',
				]
			);
			$this->end_controls_section();

			/*
			 *
			 * SETTINGS
			 */
			$this->start_controls_section(
				'section_blog_carousel_settings',
				[
					'label' => __( 'Settings', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'blog_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 4,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'blog_carousel_item_mobile',
				[
					'label'       => __( 'Mobile Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'blog_carousel_margin',
				[
					'label'   => __( 'Margin', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'blog_carousel_padding',
				[
					'label'   => __( 'Padding', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'blog_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'blog_carousel_responsive_class',
				[
					'label'   => __( 'Responsive Class', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'blog_carousel_dots',
				[
					'label'   => __( 'Dots', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'blog_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'section_posts_layout',
				[
					'label' => __( 'Layout', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);

			$this->add_control(
				'taxonomy',
				[
					'label' => __( 'Taxonomy', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'options' => get_taxonomies(),
					'description' => __( 'Select taxonomy for relative post types. If not selected then category and filter not show.', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'show_popup',
				[
					'label' => __( 'Show Popup', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => __( 'Show', 'power_elements_elementor' ),
					'label_off' => __( 'Hide', 'power_elements_elementor' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);
			$this->add_control(
				'icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => 'fa fa-plus',
					'condition'   => [
						'show_popup' => 'yes',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_filter_style',
				[
					'label' => __( 'Portfolio', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'portfolio_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_thumb' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'portfolio_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_thumb' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'portfolio_bg_color',
				[
					'label'     => __( 'Bg Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_thumb' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'portfolio_wrapper_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_portfolio_item .power_portfolio_thumb',
				]
			);
			$this->add_control(
				'portfolio_wrapper_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_thumb' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_content_wrapper_style',
				[
					'label' => __( 'Content Wrapper', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name' => 'content_wrapper_background',
					'label' => __( 'Background', 'power_elements_elementor' ),
					'types' => [ 'gradient'],
					'selector' => '{{WRAPPER}} .power_portfolio_item .power_portfolio_info',
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_icon_style',
				[
					'label' => __( 'Icon', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'icon_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_portfolio_item a.power_portfolio_popup' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'icon_size',
				[
					'label' => __( 'Size', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px'],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 50,
							'step' => 1,
						]
					],
					'default' => [
						'unit' => 'px',
						'size' => 42,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item a.power_portfolio_popup' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_content_style',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'content_options',
				[
					'label' => __( 'Info Wrapper', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_control(
				'info_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'info_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'info_bg_color',
				[
					'label'     => __( 'Bg Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'content_title_options',
				[
					'label' => __( 'Title', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_control(
				'title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_title',
				]
			);
			$this->add_control(
				'title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_title a' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'title_hover_color',
				[
					'label'     => __( 'Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_title a:hover' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'content_intro_options',
				[
					'label' => __( 'After Title', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_control(
				'af_title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_tag' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'af_title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_tag' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'af_title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_tag',
				]
			);
			$this->add_control(
				'af_title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_portfolio_item .power_portfolio_label_text .power_portfolio_item_tag' => 'color: {{VALUE}}',
					],
				]
			);
			$this->end_controls_section();
			
		}
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			//$settings = $this->get_settings();
			
			/**
			 * Setup the post arguments.
			 */
			
			$post_args              = Power_elements_Helper::power_get_post_settings( $settings );
			$query_args             = Power_elements_Helper::power_get_query_args( 'powerpost', $settings );
			$query_args             = array_merge( $query_args, $post_args, $settings );
			
			if ( isset( $query_args['tax_query'] ) ) {
				$tax_query = $query_args['tax_query'];
			}
			/**
			 * Get posts from database.
			 */
			$posts = Power_elements_Helper::power_load_portfoliocarousel( $query_args );
			/**
			 * Set total posts.
			 */
			//$total_post = $posts['count'];
			?>
			<div id="power-post-grid-<?php echo esc_attr( $this->get_id() ); ?>"
                 class="power-post-grid-container">
                <div class="power_portfolio_wrapper">
	                <div class="power_portfolio">
		                <div class="owl-carousel owl-theme power_portfoliocarousel"
		                     data-owl-carousel='{"items": <?php echo esc_attr($settings['blog_carousel_item'])?>, "margin": <?php echo esc_attr($settings['blog_carousel_margin']['size'])?>, "padding": <?php echo esc_attr($settings['blog_carousel_padding']['size'])?>, "autoplay": <?php echo esc_attr($settings['blog_carousel_autoplay'])?>, "responsiveClass": <?php echo esc_attr($settings['blog_carousel_responsive_class'])?>, "dots": <?php echo esc_attr($settings['blog_carousel_dots'])?>, "nav": <?php echo esc_attr($settings['blog_carousel_nav'])?>,"responsive":{ "0" :{ "items": <?php echo esc_attr($settings['blog_carousel_item_mobile'])?>}, "540" :{ "items": <?php echo esc_attr($settings['blog_carousel_item_mobile'])?> }, "860" :{ "items" : <?php echo esc_attr($settings['blog_carousel_item'])?> } , "1000":{ "items" : <?php echo esc_attr($settings['blog_carousel_item'])?> }}}'>
							<?php
								if ( ! empty( $posts['content'] ) ) {
									echo $posts['content'];
								} else {
									echo '<p class="text-danger">' . esc_html__( 'Something went wrong.', 'power_elements_elementor' ) . '</p>';
								}
							?>
		                </div>
	                </div>
                </div>
            </div>
			<?php
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		
		//protected function _content_template() {}
	}