<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Image_Carousel extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_image_carousel';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Image Carousel', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-media-carousel';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-image-carousel' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-image-carousel-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_image_carousel_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'power_element_carousel',
				[
					'label'      => __( 'Add Images', 'elementor' ),
					'type'       => \Elementor\Controls_Manager::GALLERY,
					'default'    => [],
					'show_label' => false,
					'dynamic'    => [
						'active' => true,
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Image_Size::get_type(),
				[
					'name'      => 'image_thumbnail',
					// Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
					'default'   => 'full',
					'separator' => 'none',
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_image_carousel_settings',
				[
					'label' => __( 'Settings', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'image_carousel_item',
				[
					'label' => __( 'Desktop Item', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::NUMBER,
					'min' => 1,
					'max' => 100,
					'step' => 1,
					'default' => 4,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'image_carousel_item_mobile',
				[
					'label' => __( 'Mobile Item', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::NUMBER,
					'min' => 1,
					'max' => 100,
					'step' => 1,
					'default' => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'image_carousel_margin',
				[
					'label' => __( 'Margin', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'range' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'image_carousel_padding',
				[
					'label' => __( 'Padding', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'range' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'image_carousel_autoplay',
				[
					'label' => __( 'Autoplay', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'image_carousel_responsive_class',
				[
					'label' => __( 'Responsive Class', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'image_carousel_dots',
				[
					'label' => __( 'Dots', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'image_carousel_nav',
				[
					'label' => __( 'Navigation', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL STYLE
			 */
			$this->start_controls_section(
				'section_image_carousel_style',
				[
					'label' => __( 'Carousel', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'image_carousel_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_imagecarousel_content' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'image_carousel_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagecarousel_content',
				]
			);
			$this->add_control(
				'image_carousel_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagecarousel_content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'image_carousel_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagecarousel_content',
				]
			);
			$this->add_control(
				'image_carousel_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagecarousel_content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'image_carousel_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagecarousel_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL ITEM STYLE
			 */
			$this->start_controls_section(
				'section_image_carousel_item_style',
				[
					'label' => __( 'Item', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'image_carousel_item_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagecarousel_content .power_imagecarousel_item',
				]
			);
			$this->add_control(
				'image_carousel_item_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagecarousel_content .power_imagecarousel_item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'image_carousel_item_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagecarousel_content .power_imagecarousel_item',
				]
			);
			$this->add_control(
				'image_carousel_item_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagecarousel_content .power_imagecarousel_item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'image_carousel_item_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagecarousel_content .power_imagecarousel_item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( empty( $settings['power_element_carousel'] ) ) {
				return;
			}
			
			$html = ' <div class="power_imagecarousel_content">
                    <div class="owl-carousel owl-theme power_imagecarousel" data-owl-carousel=\'{"items": '.esc_attr($settings['image_carousel_item']).', "margin": '.esc_attr($settings['image_carousel_margin']['size']).', "padding": '.esc_attr($settings['image_carousel_padding']['size']).', "autoplay": '.esc_attr($settings['image_carousel_autoplay']).', "responsiveClass": '.esc_attr($settings['image_carousel_responsive_class']).', "dots": '.esc_attr($settings['image_carousel_dots']).', "nav": '.esc_attr($settings['image_carousel_nav']).',"responsive":{ "0" :{ "items": '.esc_attr($settings['image_carousel_item_mobile']).' }, "540" :{ "items": '.esc_attr($settings['image_carousel_item_mobile']).' }, "860" :{ "items" : '.esc_attr($settings['image_carousel_item']).' } , "1000":{ "items" : '.esc_attr($settings['image_carousel_item']).' }}}\'>';
			foreach ( $settings['power_element_carousel'] as $index => $attachment ) {
				$image_url  = \Elementor\Group_Control_Image_Size::get_attachment_image_src( $attachment['id'], 'image_thumbnail', $settings );
				$image_html = '<img class="primary_img" src="' . esc_attr( $image_url ) . '" alt="' . esc_attr( \Elementor\Control_Media::get_image_alt( $attachment ) ) . '" />';
				
				$html .= ' <div class="item">
                            <div class="power_imagecarousel_item">
                               ' . $image_html . '
                            </div>
                        </div>';
			}
			$html .= '</div></div>';
			
			echo $html;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <div class="power_imagecarousel_content">
                <div class="owl-carousel owl-theme power_imagecarousel"
                     data-owl-carousel='{"items": {{settings.image_carousel_item}}, "margin": {{settings.image_carousel_margin.size}}, "padding": {{settings.image_carousel_padding.size}}, "autoplay": {{settings.image_carousel_autoplay}}, "responsiveClass": {{settings.image_carousel_responsive_class}}, "dots": {{settings.image_carousel_dots}}, "nav": {{settings.image_carousel_nav}},"responsive":{ "0" :{ "items": {{settings.image_carousel_item_mobile}} }, "540" :{ "items": {{settings.image_carousel_item_mobile}} }, "860" :{ "items" : {{settings.image_carousel_item}} } , "1000":{ "items" : {{settings.image_carousel_item}} }}}'>
                    <#
                    _.each( settings.power_element_carousel, function( item , i) {
                    #>
                    <div class="item">
                        <div class="power_imagecarousel_item">
                            <img class="primary_img" src="{{item.url}}" alt="">
                        </div>
                    </div>
                    <# }); #>
                </div>
            </div>
			<?php
		}
	}