<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Heading extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_heading';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Heading', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-type-tool';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-heading' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_heading',
				[
					'label' => __( 'Heading', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'heading_style',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'stp',
					'options' => [
						'stp' => __( 'S-T-P', 'power_elements_elementor' ),
						'tsp' => __( 'T-S-P', 'power_elements_elementor' )
					]
				]
			);
			$this->add_control(
				'text',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Heading', 'power_elements_elementor' ),
					'placeholder' => __( 'Title', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'text_sub',
				[
					'label'       => __( 'Subtitle', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Subtitle', 'power_elements_elementor' ),
					'placeholder' => __( 'Subtitle', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'text_para',
				[
					'label'       => __( 'Praragraph', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'default'     => __( 'Paragraph', 'power_elements_elementor' ),
					'placeholder' => __( 'Description', 'power_elements_elementor' ),
				]
			);
			$this->add_responsive_control(
				'align',
				[
					'label'   => __( 'Alignment', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::CHOOSE,
					'options' => [
						'text-left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'text-center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'text-right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'text-justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'default' => '',
				]
			);
			$this->add_control(
				'heading_css_id',
				[
					'label'       => __( 'Heading ID', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => 'power-element-heading-id',
					'title'       => __( 'Add your custom id WITHOUT the Pound key. e.g: my-id', 'power_elements_elementor' ),
					'label_block' => false,
					'description' => __( 'Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows <code>A-z 0-9</code> & underscore chars without spaces.', 'power_elements_elementor' ),
					'separator'   => 'before',
				
				]
			);
			
			$this->end_controls_section();
			
			$this->start_controls_section(
				'section_style',
				[
					'label' => __( 'Heading', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Title Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title',
				]
			);
			$this->add_control(
				'title_text_color',
				[
					'label'     => __( 'Title Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_heading_wrapper .power_title' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'power_title_margin',
				[
					'label'      => __( 'Title Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_heading_wrapper .power_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'power_title_padding',
				[
					'label'      => __( 'Title Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_heading_wrapper .power_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'title_text_shadow',
					'label'    => __( 'Title Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_heading_wrapper .power_title',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'sub_typography',
					'label'    => __( 'Subtitle Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_2,
					'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
				]
			);
			$this->add_control(
				'sub_text_color',
				[
					'label'     => __( 'Subtitle Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_heading_wrapper .power_subtitle' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'sub_title_text_shadow',
					'label'    => __( 'Subtitle Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_heading_wrapper .power_subtitle',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'para_typography',
					'label'    => __( 'Paragraph Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_3,
					'selector' => '{{WRAPPER}} .power_heading_wrapper .power_shortdesc',
				]
			);
			$this->add_control(
				'para_text_color',
				[
					'label'     => __( 'Paragraph Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_heading_wrapper .power_shortdesc' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'para_text_shadow',
					'label'    => __( 'Paragraph Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_heading_wrapper .power_shortdesc',
				]
			);
			$this->add_control(
				'social_animation',
				[
					'label'   => __( 'Animation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'animated',
					'options' => [
						'animated'     => __( 'Animation', 'power_elements_elementor' ),
						'not-animated' => __( 'No Animation', 'power_elements_elementor' )
					],
					'dynamic' => [
						'active' => true,
					]
				]
			);
			$this->add_control(
				'animation_types',
				[
					'label'     => __( 'Animation Types', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SELECT,
					'default'   => 'shake',
					'options'   => [
						'shake'      => __( 'Shake', 'power_elements_elementor' ),
						'bounce'     => __( 'Bounce', 'power_elements_elementor' ),
						'flash'      => __( 'Flash', 'power_elements_elementor' ),
						'pulse'      => __( 'Pulse', 'power_elements_elementor' ),
						'swing'      => __( 'Swing', 'power_elements_elementor' ),
						'tada'       => __( 'Tada', 'power_elements_elementor' ),
						'wobble'     => __( 'Wobble', 'power_elements_elementor' ),
						'jello'      => __( 'Jello', 'power_elements_elementor' ),
						'zoomIn'     => __( 'Zoom In', 'power_elements_elementor' ),
						'heartBeat'  => __( 'Heart Beat', 'power_elements_elementor' ),
						'fadeIn'     => __( 'Fade In', 'power_elements_elementor' ),
						'rubberBand' => __( 'Rubber Band', 'power_elements_elementor' ),
					],
					'condition' => [
						'social_animation' => 'animated',
					],
					'dynamic'   => [
						'active' => true,
					]
				]
			);
			
			
			$this->add_control(
				'background_color',
				[
					'label'     => __( 'Divider Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_heading_wrapper:after' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'heading_box_shadow',
					'selector' => '{{WRAPPER}} .power_heading_wrapper',
				]
			);
			$this->add_control(
				'power_heading_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_heading_wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'power_heading_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_heading_wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$this->add_render_attribute( 'wrapper', 'class', [
				'power_heading_wrapper',
				$settings['animation_types'],
				$settings['social_animation'],
				$settings['align']
			] );
			
			$title    = ( $settings['text'] != '' ) ? '<h2 class="power_title">' . esc_html( $settings['text'] ) . '</h2>' : '';
			$subtitle = ( $settings['text_sub'] != '' ) ? '<h5 class="power_subtitle">' . esc_html( $settings['text_sub'] ) . '</h5>' : '';
			$para     = ( trim($settings['text_para']) != '' ) ? '<p class="power_shortdesc">' . esc_html( $settings['text_para'] ) . '</p>' : '';
			
			if ( $settings['heading_style'] == 'stp' ) {
				$html = $subtitle . $title . $para;
			} else {
				$html = $title . $subtitle . $para;
			}
			echo '<div ' . $this->get_render_attribute_string( 'wrapper' ) . '>
	    		' . $html . '
	    	</div>';
			
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#
            view.addRenderAttribute( 'wrapper', {
            'class': [ 'power_heading_wrapper', settings.animation_types, settings.social_animation,  settings.align]
            } );
            
            #>
            <div class="elementor-social-wrapper">
                <div {{{ view.getRenderAttributeString(
                'wrapper' ) }}}>
                <# if ( settings.heading_style=='stp' ) { #>
                <h5 class="power_subtitle">{{{ settings.text_sub }}}</h5>
                <h2 class="power_title">{{{ settings.text }}}</h2>
                <p class="power_shortdesc">{{{ settings.text_para }}}</p>
                <# }else{ #>
                <h2 class="power_title">{{{ settings.text }}}</h2>
                <h5 class="power_subtitle">{{{ settings.text_sub }}}</h5>
                <p class="power_shortdesc">{{{ settings.text_para }}}</p>
                <# } #>
            </div>
            </div>
			<?php
			
		}
	}