<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Circle_Progressbar extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_circle_progress';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Circle Progressbar', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-counter-circle';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-circle-progressbar' ];
		}
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-circle-progressbar-js' ];
		}
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_progress_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'progress_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Marketing', 'power_elements_elementor' ),
					'placeholder' => __( 'Title', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'progress_value',
				[
					'label'       => __( 'Value', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'default'     => __( '65', 'power_elements_elementor' ),
					'placeholder' => __( 'Value.', 'power_elements_elementor' ),
					'description' => __( 'Value is numerical.', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'progress_value_postfix',
				[
					'label'       => __( 'Postfix', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( '%', 'power_elements_elementor' ),
					'description' => __( 'Postfix of value field. if value is 85 then % will be its postfix. 85%', 'power_elements_elementor' ),
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR
			 */
			$this->start_controls_section(
				'power_element_progress_wrapper_style',
				[
					'label' => __( 'Circular', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'align',
				[
					'label'        => __( 'Alignment', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular_content' => 'text-align: {{VALUE}}',
					]
				]
			);
			$this->add_control(
				'progress_width',
				[
					'label' => __( 'Width', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
							'step' => 5,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 200,
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular' => 'width: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_control(
				'progress_height',
				[
					'label' => __( 'Height', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
							'step' => 5,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 200,
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular' => 'height: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_control(
				'progress_fill_color',
				[
					'label'     => __( 'Fill Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_track' => 'fill: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'progress_stroke_color',
				[
					'label'     => __( 'Stroke Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_track' => 'stroke: {{VALUE}}',
					]
				]
			);
			$this->add_control(
				'progress_stroke_width',
				[
					'label' => __( 'Stroke Width', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 50,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 5,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 8,
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_track' => 'stroke-width: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR CIRCLE MOVE
			 */
			$this->start_controls_section(
				'power_element_progress_circle_style',
				[
					'label' => __( 'Circle Move', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'progress_circle_stroke_width',
				[
					'label' => __( 'Stroke Width', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 50,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 5,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 4,
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_fill' => 'stroke-width: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_control(
				'progress_circle_fill_color',
				[
					'label'     => __( 'Fill Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_fill' => 'fill: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'progress_circle_stroke_color',
				[
					'label'     => __( 'Stroke Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_fill' => 'stroke: {{VALUE}}',
					]
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR CIRCLE MOVE
			 */
			$this->start_controls_section(
				'power_element_progress_value_style',
				[
					'label' => __( 'Value', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'progress_value_fill_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_value' => 'fill: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'progress_value_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_progress_circular .power_progress_circular_value',
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR CIRCLE MOVE
			 */
			$this->start_controls_section(
				'power_element_progress_text_style',
				[
					'label' => __( 'Text', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'progress_text_fill_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_circular .power_progress_circular_text' => 'fill: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'progress_text_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_progress_circular .power_progress_circular_text',
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$value = $settings['progress_value']!=''? $settings['progress_value'].$settings['progress_value_postfix'] : '0';
			$this->add_render_attribute( 'circle_postfix', 'data-postfix', $settings['progress_value_postfix']);
			echo '<div class="power_progress_circular_content">
                <svg class="power_progress_circular" '.$this->get_render_attribute_string( 'circle_postfix' ).' data-progress="'.esc_attr($settings['progress_value']).'" x="0px" y="0px" viewBox="0 0 80 80">
                    <path class="power_progress_circular_track" d="M5,40a35,35 0 1,0 70,0a35,35 0 1,0 -70,0" />
                    <path class="power_progress_circular_fill" d="M5,40a35,35 0 1,0 70,0a35,35 0 1,0 -70,0" />
                    <text class="power_progress_circular_value" x="50%" y="45%">'.esc_html($value).'</text>
                    <text class="power_progress_circular_text" x="50%" y="65%">'.esc_html($settings['progress_title']).'</text>
                </svg>
            </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#
            view.addRenderAttribute( 'wrapper', {
            'data-postfix':settings.progress_value_postfix,
            'class':'power_progress_circular',
            'data-progress':settings.progress_value,
            'x':'0px',
            'y':'0px',
            'viewBox':'0 0 80 80'
            } );

            #>
            <div class="power_progress_circular_content">
                <svg {{{ view.getRenderAttributeString(
                'wrapper' ) }}}>
                    <path class="power_progress_circular_track" d="M5,40a35,35 0 1,0 70,0a35,35 0 1,0 -70,0" />
                    <path class="power_progress_circular_fill" d="M5,40a35,35 0 1,0 70,0a35,35 0 1,0 -70,0" />
                    <text class="power_progress_circular_value" x="50%" y="45%">{{{settings.progress_value}}}</text>
                    <text class="power_progress_circular_text" x="50%" y="65%">{{{settings.progress_title}}}</text>
                </svg>
            </div>
			<?php
		}
	}