<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Team_Carousel extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_team_carousel';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Team Carousel', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-lock-user';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-team-carousel' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-teamcarousel-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_team_content',
				[
					'label' => __( 'Team Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'team_style',
				[
					'label'   => __( 'Style', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'style_1',
					'options' => [
						'style_1' => __( 'Style 1', 'power_elements_elementor' ),
						'style_2' => __( 'Style 2', 'power_elements_elementor' ),
						'style_3' => __( 'Style 3', 'power_elements_elementor' )
					]
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'name_text',
				[
					'label'       => __( 'Name', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Your Name', 'power_elements_elementor' ),
					'placeholder' => __( 'Your Name Please', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'designation_text',
				[
					'label'       => __( 'Address or Designation', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'CEO, Advertising Corp', 'power_elements_elementor' ),
					'placeholder' => __( 'Give your designation please.', 'power_elements_elementor' ),
					'label_block' => true
				]
			);
			$repeater->add_control(
				'description_text',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'default'     => __( '', 'power_elements_elementor' ),
					'placeholder' => __( 'Short description.', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'team_image',
				[
					'label'   => __( 'Choose Image', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			$repeater->add_control(
				'team_social_options',
				[
					'label'   => __( 'Show/Hide', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'show',
					'options' => [
						'show' => __( 'Show', 'power_elements_elementor' ),
						'hide' => __( 'Hide', 'power_elements_elementor' )
					]
				]
			);
			$this->add_control(
				'list',
				[
					'label'       => __( 'Team List', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'list_title'   => __( 'Title #1', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
						[
							'list_title'   => __( 'Title #2', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
					],
					'title_field' => '{{{ name_text }}}',
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL SETTINGS
			 */
			$this->start_controls_section(
				'section_team_carousel_settings',
				[
					'label' => __( 'Settings', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'team_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 4,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'team_carousel_item_mobile',
				[
					'label'       => __( 'Mobile Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'team_carousel_margin',
				[
					'label'   => __( 'Margin', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'team_carousel_padding',
				[
					'label'   => __( 'Padding', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'team_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'team_carousel_responsive_class',
				[
					'label'   => __( 'Responsive Class', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'team_carousel_dots',
				[
					'label'   => __( 'Dots', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'team_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'false',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE
			 */
			$this->start_controls_section(
				'power_element_style',
				[
					'label' => __( 'Image Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'team_img_background',
					'label'    => __( 'Background', 'power_elements_elementor' ),
					'types'    => [ 'classic', 'gradient', 'video' ],
					'selector' => '{{WRAPPER}} .power_team_member_image',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'team_img_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_image',
				]
			);
			$this->add_control(
				'team_img_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'team_img_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'more_options',
				[
					'label'     => __( 'Overlay', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'       => 'team_img_overlay_background',
					'label'      => __( 'Background Overlay', 'power_elements_elementor' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .power_team_member.simple_style .power_team_member_image:before',
				]
			);
			$this->end_controls_section();
			/*
			 *
			 * DETAILS STYLE
			 */
			
			$this->start_controls_section(
				'power_element_details_style',
				[
					'label' => __( 'Details Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'team_details_align',
				[
					'label'     => __( 'Alignment', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_details'                                        => 'text-align: {{VALUE}};',
						'{{WRAPPER}} .power_team_member.overlay_style .power_team_member_intro'                            => 'text-align: {{VALUE}};',
						'{{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'       => 'team_details_background',
					'label'      => __( 'Background', 'power_elements_elementor' ),
					'types'      => [ 'classic', 'gradient' ],
					'show_label' => true,
					'selector'   => '{{WRAPPER}} .power_team_member .power_team_member_details, {{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner',
				]
			);
			$this->add_control(
				'team_details_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'team_details_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details, {{WRAPPER}} .power_team_member.overlay_style .power_team_member_details .power_team_member_inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Name Style
			 */
			$this->start_controls_section(
				'power_element_name_style',
				[
					'label' => __( 'Name Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_title',
				]
			);
			$this->add_control(
				'name_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'name_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_title',
				]
			);
			$this->add_control(
				'name_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'name_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Designation Style
			 */
			$this->start_controls_section(
				'power_element_designation_style',
				[
					'label' => __( 'Designation Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'designation_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_designation',
				]
			);
			$this->add_control(
				'designation_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_designation' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'designation_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_designation',
				]
			);
			$this->add_control(
				'designation_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_designation' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'designation_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_designation' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Description Style
			 */
			$this->start_controls_section(
				'power_element_des_style',
				[
					'label' => __( 'Description Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_text' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( $settings['list'] ) {
				$html = '<div class="owl-carousel owl-theme power_teamcarousel" data-owl-carousel=\'{"items": ' . esc_attr( $settings['team_carousel_item'] ) . ', "margin": ' . esc_attr( $settings['team_carousel_margin']['size'] ) . ', "padding": ' . esc_attr( $settings['team_carousel_padding']['size'] ) . ', "autoplay": ' . esc_attr( $settings['team_carousel_autoplay'] ) . ', "responsiveClass": ' . esc_attr( $settings['team_carousel_responsive_class'] ) . ', "dots": ' . esc_attr( $settings['team_carousel_dots'] ) . ', "nav": ' . esc_attr( $settings['team_carousel_nav'] ) . ',"responsive":{ "0" :{ "items": ' . esc_attr( $settings['team_carousel_item_mobile'] ) . ' }, "540" :{ "items": ' . esc_attr( $settings['team_carousel_item_mobile'] ) . ' }, "860" :{ "items" : ' . esc_attr( $settings['team_carousel_item'] ) . ' } , "1000":{ "items" : ' . esc_attr( $settings['team_carousel_item'] ) . ' }}}\'>';
				foreach ( $settings['list'] as $item ) {
					if ( $item['team_image']['id'] != '' ) {
						$img = '<div class="power_team_member_image">
                            <img src="' . esc_url( $item['team_image']['url'] ) . '" alt="' . esc_attr( $item['name_text'] ) . '">
                        </div>';
					} else {
						$img = '';
					}
					
					if ( $item['description_text'] != '' ) {
						$description = '<p class="power_team_member_text">' . esc_html( $item['description_text'] ) . '</p>';
					} else {
						$description = '';
					}
					
					if ( $settings['team_style'] == 'style_2' ) {
						$html .= ' <div class="item"><div id="power_team_member_2" class="power_team_member overlay_style">
					                 ' . $img . '
					                <div class="power_team_member_intro">
					                     <h4 class="power_team_member_intro_title">' . esc_html( $item['name_text'] ) . '</h3>
						                 <h6 class="power_team_member_intro_designation">' . esc_html( $item['designation_text'] ) . '</h5>
					                </div>
					                <div class="power_team_member_details">
					                    <div class="power_team_member_inner">
					                         <h3 class="power_team_member_title">' . esc_html( $item['name_text'] ) . '</h3>
						                     <h5 class="power_team_member_designation">' . esc_html( $item['designation_text'] ) . '</h5>
					                        ' . $description . '
					                    </div>
					                </div>
					            </div></div>';
					} elseif ( $settings['team_style'] == 'style_3' ) {
						$html .= ' <div class="item"><div id="power_team_member_3" class="power_team_member overlay_style">
					               ' . $img . '
					                <div class="power_team_member_details">
					                    <div class="power_team_member_inner">
					                         <h3 class="power_team_member_title">' . esc_html( $item['name_text'] ) . '</h3>
						                     <h5 class="power_team_member_designation">' . esc_html( $item['designation_text'] ) . '</h5>
					                        ' . $description . '
					                    </div>
					                </div>
					            </div></div>';
					} else {
						$html .= ' <div class="item"><div id="power_team_member_1" class="power_team_member simple_style">
				                ' . $img . '
					                <div class="power_team_member_details">
					                    <div class="power_team_member_inner">
					                        <h3 class="power_team_member_title">' . esc_html( $item['name_text'] ) . '</h3>
					                        <h5 class="power_team_member_designation">' . esc_html( $item['designation_text'] ) . '</h5>
					                        ' . $description . '
					                    </div>
					                </div>
				                </div></div>';
					}
				}
				$html .= '</div>';
				
				echo $html;
			} else {
				echo esc_html__( 'Something wrong', 'power_elements_elementor' );
			}
			
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <div class="owl-carousel owl-theme power_imagecarousel"
                 data-owl-carousel='{"items": {{settings.team_carousel_item}}, "margin": {{settings.team_carousel_margin.size}}, "padding": {{settings.team_carousel_padding.size}}, "autoplay": {{settings.team_carousel_autoplay}}, "responsiveClass": {{settings.team_carousel_responsive_class}}, "dots": {{settings.team_carousel_dots}}, "nav": {{settings.team_carousel_nav}},"responsive":{ "0" :{ "items": {{settings.team_carousel_item_mobile}} }, "540" :{ "items": {{settings.team_carousel_item_mobile}} }, "860" :{ "items" : {{settings.team_carousel_item}} } , "1000":{ "items" : {{settings.team_carousel_item}} }}}'>
                <# _.each( settings.list, function( item ) { #>
                <# if ( settings.team_style == 'style_2' ) { #>
                <div id="power_team_member_2" class="power_team_member overlay_style">
                    <# if ( item.team_image.id != '' ) { #>
                    <div class="power_team_member_image">
                        <img src="{{item.team_image.url}}" alt="{{item.name_text}}">
                    </div>
                    <# } #>
                    <div class="power_team_member_intro">
                        <h4 class="power_team_member_intro_title">{{{item.name_text}}}</h4>
                        <h6 class="power_team_member_intro_designation">{{{item.designation_text}}}</h6>
                    </div>
                    <div class="power_team_member_details">
                        <div class="power_team_member_inner">
                            <h3 class="power_team_member_title">{{{item.name_text}}}</h3>
                            <h5 class="power_team_member_designation">{{{item.designation_text}}}</h5>
                            <# if ( item.description_text != '' ) { #>
                            <p class="power_team_member_text">{{{item.description_text}}}</p>
                            <# } #>
                        </div>
                    </div>
                </div>
                <# } else if(settings.team_style == 'style_3'){ #>
                <div id="power_team_member_3" class="power_team_member overlay_style">
                    <# if ( item.team_image.id != '' ) { #>
                    <div class="power_team_member_image">
                        <img src="{{item.team_image.url}}" alt="{{item.name_text}}">
                    </div>
                    <# } #>
                    <div class="power_team_member_details">
                        <div class="power_team_member_inner">
                            <h3 class="power_team_member_title">{{{item.name_text}}}</h3>
                            <h5 class="power_team_member_designation">{{{item.designation_text}}}</h5>
                            <# if ( item.description_text != '' ) { #>
                            <p class="power_team_member_text">{{{item.description_text}}}</p>
                            <# } #>
                        </div>
                    </div>
                </div>
                <# }else{#>
                <div id="power_team_member_1" class="power_team_member simple_style">
                    <# if ( item.team_image.id != '' ) { #>
                    <div class="power_team_member_image">
                        <img src="{{item.team_image.url}}" alt="{{item.name_text}}">
                    </div>
                    <# } #>
                    <div class="power_team_member_details">
                        <div class="power_team_member_inner">
                            <h3 class="power_team_member_title">{{{item.name_text}}}</h3>
                            <h5 class="power_team_member_designation">{{{item.designation_text}}}</h5>
                        </div>
                    </div>
                </div>
                <# } #>
                <# }); #>
            </div>
			<?php
		}
	}