<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Slider extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_slider';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Slider', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-media-carousel';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-slider', 'power-element-swiper' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-slider-js'];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_slider_content',
				[
					'label' => __( 'Slider Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'slider_style',
				[
					'label'   => __( 'Style', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'slide_1',
					'options' => [
						'slide_1' => __( 'Style 1', 'power_elements_elementor' ),
						'slide_2' => __( 'Style 2', 'power_elements_elementor' ),
						'slide_3' => __( 'Style 3', 'power_elements_elementor' ),
					]
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'sub_heading',
				[
					'label'       => __( 'Slider Tag', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Slider Tag', 'power_elements_elementor' ),
					'placeholder' => __( '', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'heading',
				[
					'label'       => __( 'Sub Heading', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Slider Sub Heading', 'power_elements_elementor' ),
					'placeholder' => __( '', 'power_elements_elementor' ),
					'label_block' => true
				]
			);
			$repeater->add_control(
				'description_text',
				[
					'label'       => __( 'Heading', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'default'     => __( 'Slider Heading', 'power_elements_elementor' ),
					'placeholder' => __( '', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'slider_image',
				[
					'label'   => __( 'Choose Image', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			$repeater->add_control(
				'button_show_hide',
				[
					'label'        => __( 'Button', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SWITCHER,
					'label_on'     => __( 'show', 'power_elements_elementor' ),
					'label_off'    => __( 'Hide', 'power_elements_elementor' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$repeater->add_control(
				'link',
				[
					'label'       => __( 'Button Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'dynamic'     => [
						'active' => true,
					],
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'default'     => [
						'url' => '#',
					],
					'condition'   => [
						'button_show_hide' => 'yes',
					],
				]
			);
			$repeater->add_control(
				'button_text',
				[
					'label'       => __( 'Button Text', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true,
					],
					'default'     => __( 'Button Text', 'power_elements_elementor' ),
					'placeholder' => __( 'Read More', 'power_elements_elementor' ),
					'condition'   => [
						'button_show_hide' => 'yes',
					]
				]
			);
			$repeater->add_control(
				'button_style',
				[
					'label'        => __( 'Type', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SELECT,
					'default'      => 'power_slider_button effect_1',
					'options'      => [
						'power_slider_button effect_1'  => __( 'Effect 1', 'power_elements_elementor' ),
						'power_slider_button effect_2'  => __( 'Effect 2', 'power_elements_elementor' ),
						'power_slider_button effect_3'  => __( 'Effect 3', 'power_elements_elementor' ),
						'power_slider_button effect_4'  => __( 'Effect 4', 'power_elements_elementor' ),
						'power_slider_button effect_5'  => __( 'Effect 5', 'power_elements_elementor' ),
						'power_slider_button effect_6'  => __( 'Effect 6', 'power_elements_elementor' ),
						'power_slider_button effect_7'  => __( 'Effect 7', 'power_elements_elementor' ),
						'power_slider_button effect_8'  => __( 'Effect 8', 'power_elements_elementor' ),
						'power_slider_button effect_9'  => __( 'Effect 9', 'power_elements_elementor' ),
					]
				]
			);
			$this->add_control(
				'list',
				[
					'label'       => __( 'Slider List', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'heading'   => __( 'Incredible Iceland', 'power_elements_elementor' ),
							'sub_heading' => __( 'Travelling', 'power_elements_elementor' ),
						],
						[
							'heading'   => __( 'Beautiful Nature', 'power_elements_elementor' ),
							'sub_heading' => __( 'Travelling', 'power_elements_elementor' ),
						],
					],
					'title_field' => '{{{ heading }}}',
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL SETTINGS
			 */
			$this->start_controls_section(
				'section_slider_carousel_settings',
				[
					'label' => __( 'Settings', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'slider_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'slider_carousel_margin',
				[
					'label'   => __( 'Margin', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'slider_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'slider_carousel_loop',
				[
					'label'   => __( 'Loop', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'slider_carousel_dots',
				[
					'label'   => __( 'Dots', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'slider_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE
			 */
			$this->start_controls_section(
				'power_element_style',
				[
					'label' => __( 'Slider Content Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'slider_img_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_item',
				]
			);
			$this->add_control(
				'slider_img_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'slider_img_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 *
			 * DETAILS STYLE
			 */
			
			$this->start_controls_section(
				'power_element_alignment',
				[
					'label' => __( 'Content Alignment', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'slider_details_align',
				[
					'label'     => __( 'Alignment', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors' => [
						'{{WRAPPER}} .slider-details' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * Slider Tag Style
			 */
			$this->start_controls_section(
				'power_element_des_style',
				[
					'label' => __( 'Slider Tag Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_slider_item .slider-details .slider-tag',
				]
			);
			$this->add_control(
				'des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_slider_item .slider-details .slider-tag' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_item .slider-details .slider-tag',
				]
			);
			$this->add_control(
				'des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item .slider-details .slider-tag' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item .slider-details .slider-tag' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Sub Heading Style
			 */
			$this->start_controls_section(
				'power_element_sub_heading_style',
				[
					'label' => __( 'Sub Heading Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slider-details .slide_heading .sub_heading',
				]
			);
			$this->add_control(
				'sub_heading_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slider-details .slide_heading .sub_heading' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'sub_heading_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .slider-details .slide_heading .sub_heading',
				]
			);
			$this->add_control(
				'sub_heading_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider-details .slide_heading .sub_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'sub_heading_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider-details .slide_heading .sub_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Heading Style
			 */
			$this->start_controls_section(
				'power_element_heading_style',
				[
					'label' => __( 'Heading Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'heading_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slider-details .slide_heading .default_heading',
				]
			);
			$this->add_control(
				'heading_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slider-details .slide_heading .default_heading' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'heading_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .slider-details .slide_heading .default_heading',
				]
			);
			$this->add_control(
				'heading_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider-details .slide_heading .default_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'heading_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slider-details .slide_heading .default_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * STYLE BUTTON
			 */
			$this->start_controls_section(
				'section_button_style',
				[
					'label' => __( 'Button', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'button_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_hover_color',
				[
					'label'     => __( 'Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:hover' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'button_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			$this->add_control(
				'button_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:after' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_bg_hover_color',
				[
					'label'     => __( 'Hover Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:before' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_bg_hidden_color',
				[
					'label'     => __( 'Hidden Bg Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'button_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'button_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			$this->add_control(
				'button_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'button_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button_group' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'button_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( $settings['list'] ) { ?>

				<?php $html = '<div class="'.esc_html( $settings['slider_style'] ).'">
					<div class="swiper-container" 
						data-swiper_limit="' . esc_attr( $settings['slider_carousel_item'] ) . '"
						data-swiper_autoplay="' . esc_attr( $settings['slider_carousel_autoplay'] ) . '"
						data-swiper_margin="' . esc_attr( $settings['slider_carousel_margin']['size'] ) . '"
						data-swiper_loop="' . esc_attr( $settings['slider_carousel_loop'] ) . '"
						>
						<div class="swiper-wrapper">';
				
				
				
				foreach ( $settings['list'] as $item ) {

					if ( $item['button_show_hide']=='yes' ) {
					if ( ! empty( $item['link'] ) ) {
						$this->add_render_attribute( 'button', 'href', $item['link'] );
					}

					


					$button_html = '<div class="button_group">
										<a  ' . $this->get_render_attribute_string( 'button' ) . ' class="button '.esc_html( $item['button_style'] ).' fade_black_button" data-text="' . esc_html( $item['button_text'] ) . '"><span>' . esc_html( $item['button_text'] ) . '</span></a>
									</div>';
				} else {
					$button_html = '';
				}
					if ( $item['slider_image']['id'] != '' ) {
						$img = 'style="background-image:url(' . esc_url( $item['slider_image']['url'] ) . ')"';
					} else {
						$img = '';
					}
					
					if ( $item['description_text'] != '' ) {
						$description = esc_html( $item['description_text'] ) ;
					} else {
						$description = '';
					}
					
					if ( $settings['slider_style'] == 'slide_2' ) {
						$html .= ' <div class="swiper-slide"> 
										<div class="slider_item" ' . $img . '>
											<div class="slider-inner">
												<div class="slider-details">
													<h1 class="slide_heading"><span>' . esc_html( $item['heading'] ) . '</span></h1>
													<h4 class="slider-tag">' . esc_html( $item['sub_heading'] ) . '</h4>
												</div>
												<a class="popup fancyswiper" href="'.esc_url( $item['slider_image']['url'] ).'" data-fancybox="images">
													<img src="'.PLUGIN_DIRECTORY_URL.'/assets/img/g-icon2.png" alt="icon">
												</a>
											</div>
										</div>
									</div>';
					} elseif ( $settings['slider_style'] == 'slide_3' ) {
						$html .= ' <div class="swiper-slide">
						<div class="slider_item" ' . $img . '>
							<div class="slider-inner">
								<div class="slider-details">
									<h4 class="slider-tag">' . esc_html( $item['sub_heading'] ) . '</h4>
									<h1 class="slide_heading"><span>' . esc_html( $item['heading'] ) . '</span></h1>
									' . $button_html . '
								</div>
								<a class="popup fancyswiper" href="'.esc_url( $item['slider_image']['url'] ).'" data-fancybox="images">
									<img src="'.PLUGIN_DIRECTORY_URL.'/assets/img/g-icon2.png" alt="icon">
								</a>
							</div>
						</div>
					</div>';
					} else {
						$html .= ' <div class="swiper-slide">
										<div class="slider_item"  ' . $img . '>
											<div class="slider-inner">
												<div class="slider-details">
													<h4 class="slider-tag">' . esc_html( $item['sub_heading'] ) . '</h4>
													<h1 class="slide_heading"><span>' . esc_html( $item['heading'] ) . '</span></h1>
													' . $button_html . '
												</div>
												<a class="popup fancyswiper" href="'.esc_url( $item['slider_image']['url'] ).'" data-fancybox="images">
													<img src="'.PLUGIN_DIRECTORY_URL.'/assets/img/g-icon2.png" alt="icon">
												</a>
											</div>
										</div>
									</div>';
					}
				}
				if ( $settings['slider_style'] == 'slide_2' ) {
					$html .= ' </div>
							</div>
							<div class="swiper-controller">
								<div class="swiper-scrollbar"></div>
							</div>
						</div>';
				}else{
					$html .= '</div>
						</div>
						<div class="swiper-controller">
							<div class="swiper-pagination"></div>
							<div class="swiper-navigation">
								<div class="swiper-button-prev"></div>
								<div class="swiper-button-next"></div>
							</div>
						</div>
					</div>';

				}

				echo $html;
			} else {
				echo esc_html__( 'Something wrong', 'power_elements_elementor' );
			}
			
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}