<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Map extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_map';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Map', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-google-maps';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-map' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-map-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_map_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'power_map_lat',
				[
					'label'       => __( 'Latitude', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'default'     => __( '51.508742', 'power_elements_elementor' ),
					'placeholder' => __( '51.508742', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'power_map_long',
				[
					'label'       => __( 'Longitude', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'default'     => __( '-0.120850', 'power_elements_elementor' ),
					'placeholder' => __( '-0.120850', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'power_map_zoom',
				[
					'label'   => __( 'Zoom', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::NUMBER,
					'min'     => 1,
					'max'     => 20,
					'step'    => 1,
					'default' => 14,
				]
			);
			$this->add_control(
				'power_map_scrollwheel',
				[
					'label'   => __( 'Scroll wheel', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => true,
					'options' => [
						true  => __( 'True', 'power_elements_elementor' ),
						false => __( 'False', 'power_elements_elementor' )
					]
				]
			);
			$this->add_control(
				'power_map_draggable',
				[
					'label'   => __( 'Draggable', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => true,
					'options' => [
						true  => __( 'True', 'power_elements_elementor' ),
						false => __( 'False', 'power_elements_elementor' )
					]
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_map_stylet',
				[
					'label' => __( 'Map', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'map_background_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_googlemap_content' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'map_height',
				[
					'label' => __( 'Height', 'plugin-domain' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
							'step' => 5,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 450,
					],
					'selectors' => [
						'{{WRAPPER}} .power_googlemap_content .power_googlemap' => 'height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'map_box_shadow',
					'selector' => '{{WRAPPER}} .power_googlemap_content',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'      => 'map_border',
					'selector'  => '{{WRAPPER}} .power_googlemap_content',
					'separator' => 'before',
				]
			);
			
			$this->add_control(
				'map_border_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}}.power_googlemap_content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'map_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .power_googlemap_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->add_responsive_control(
				'map_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .power_googlemap_content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			$this->add_render_attribute( 'google_map', 'data-lat', $settings['power_map_lat']);
			$this->add_render_attribute( 'google_map', 'data-long', $settings['power_map_long']);
			$this->add_render_attribute( 'google_map', 'data-zoom', $settings['power_map_zoom']);
			$this->add_render_attribute( 'google_map', 'data-scroll', $settings['power_map_scrollwheel']);
			$this->add_render_attribute( 'google_map', 'data-drag', $settings['power_map_draggable']);
			echo '<div class="power_googlemap_content">
                <div id="googleMap" class="power_googlemap" '.$this->get_render_attribute_string( 'google_map' ).'></div>
            </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
			<#
			view.addRenderAttribute( 'google_map', {
			'data-lat': settings.power_map_lat,
			'data-long': settings.power_map_long,
			'data-zoom': settings.power_map_zoom,
			'data-scroll': settings.power_map_scrollwheel,
			'data-drag': settings.power_map_draggable,
			} );
			#>
			<div class="power_googlemap_content">
				<div id="googleMap" class="power_googlemap" {{{ view.getRenderAttributeString( 'google_map' ) }}}></div>
			</div>
			<?php
		}
	}