<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Icon_List extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_icon_list';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Icon List', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-bullet-list';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-icon-list' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_list',
				[
					'label' => __( 'Icon List', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'view',
				[
					'label'          => __( 'Layout', 'power_elements_elementor' ),
					'type'           => \Elementor\Controls_Manager::CHOOSE,
					'default'        => 'block',
					'options'        => [
						'block'        => [
							'title' => __( 'Default', 'power_elements_elementor' ),
							'icon'  => 'eicon-editor-list-ul',
						],
						'inline-items' => [
							'title' => __( 'Inline', 'power_elements_elementor' ),
							'icon'  => 'eicon-ellipsis-h',
						],
					],
					'render_type'    => 'template',
					'style_transfer' => true,
				]
			);
			
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'link',
				[
					'label'       => __( 'Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'dynamic'     => [
						'active' => true,
					],
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'default'     => [
						'url' => '#',
					],
				]
			);
			$repeater->add_control(
				'text',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true,
					],
					'default'     => __( 'Heading', 'power_elements_elementor' ),
					'placeholder' => __( 'Title', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
				]
			);
			$repeater->add_control(
				'icon_align',
				[
					'label'       => __( 'Icon Position', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::SELECT,
					'default'     => 'left',
					'options'     => [
						'right' => __( 'Right', 'power_elements_elementor' ),
						'left'  => __( 'Left', 'power_elements_elementor' ),
					],
					'condition'   => [
						'icon!' => '',
					],
					'render_type' => 'template',
				]
			);
			
			$this->add_control(
				'icon_list',
				[
					'label'       => '',
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'text' => __( 'List Item #1', 'power_elements_elementor' ),
							'icon' => 'fa fa-check',
						],
						[
							'text' => __( 'List Item #2', 'power_elements_elementor' ),
							'icon' => 'fa fa-times',
						],
						[
							'text' => __( 'List Item #3', 'power_elements_elementor' ),
							'icon' => 'fa fa-dot-circle-o',
						],
					],
					'title_field' => '<i class="{{ icon }}" aria-hidden="true"></i> {{{ text }}}',
				]
			);
			
			$this->add_control(
				'icon_list_css_id',
				[
					'label'       => __( 'Heading ID', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true,
					],
					'default'     => 'power-element-icon-list-id',
					'title'       => __( 'Add your custom id WITHOUT the Pound key. e.g: my-id', 'power_elements_elementor' ),
					'label_block' => false,
					'description' => __( 'Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows <code>A-z 0-9</code> & underscore chars without spaces.', 'power_elements_elementor' ),
					'separator'   => 'before',
				
				]
			);
			
			$this->end_controls_section();
			
			$this->start_controls_section(
				'section_style',
				[
					'label' => __( 'List', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'iconlist_background',
					'label'    => __( 'Background', 'power_elements_elementor' ),
					'types'    => [ 'classic', 'gradient' ],
					'selector' => '{{WRAPPER}} .power_iconilist .power_iconilist_item',
				]
			);
			$this->add_responsive_control(
				'iconlist_padding',
				[
					'label'      => __( 'padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'iconlist_margin',
				[
					'label'      => __( 'margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->add_responsive_control(
				'align',
				[
					'label'     => __( 'Alignment', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::CHOOSE,
					'options'   => [
						'align-left'   => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'eicon-h-align-left',
						],
						'align-center' => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'eicon-h-align-center',
						],
						'align-right'  => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'eicon-h-align-right',
						]
					],
					'default'   => 'align-left',
					'condition' => [
						'view' => 'block',
					]
				]
			);
			
			$this->add_control(
				'divider',
				[
					'label'     => __( 'Divider', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'label_off' => __( 'Off', 'power_elements_elementor' ),
					'label_on'  => __( 'On', 'power_elements_elementor' ),
					'selectors' => [
						'{{WRAPPER}} .power_iconilist.block .power_iconilist_item:not(:last-child)'        => 'border-bottom:1px dotted red;',
						'{{WRAPPER}} .power_iconilist.inline-items .power_iconilist_item:not(:last-child)' => 'border-right:1px dotted orange;',
					],
					'separator' => 'before',
				]
			);
			
			$this->add_control(
				'divider_style',
				[
					'label'     => __( 'Style', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SELECT,
					'options'   => [
						'solid'  => __( 'Solid', 'power_elements_elementor' ),
						'double' => __( 'Double', 'power_elements_elementor' ),
						'dotted' => __( 'Dotted', 'power_elements_elementor' ),
						'dashed' => __( 'Dashed', 'power_elements_elementor' ),
					],
					'default'   => 'solid',
					'condition' => [
						'divider' => 'yes',
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconilist.block .power_iconilist_item:not(:last-child)'        => 'border-bottom-style:{{VALUE}};',
						'{{WRAPPER}} .power_iconilist.inline-items .power_iconilist_item:not(:last-child)' => 'border-right-style:{{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'divider_weight',
				[
					'label'     => __( 'Weight', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'default'   => [
						'size' => 1,
					],
					'range'     => [
						'px' => [
							'min' => 1,
							'max' => 20,
						],
					],
					'condition' => [
						'divider' => 'yes',
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconilist.block .power_iconilist_item:not(:last-child)'        => 'border-bottom-width:{{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .power_iconilist.inline-items .power_iconilist_item:not(:last-child)' => 'border-right-width:{{SIZE}}{{UNIT}};',
					],
				]
			);
			
			$this->add_control(
				'divider_width',
				[
					'label'     => __( 'Width', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'default'   => [
						'unit' => '%',
					],
					'condition' => [
						'divider' => 'yes',
						'view!'   => 'inline-items',
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconilist.block .power_iconilist_item:not(:last-child)' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			
			$this->add_control(
				'divider_height',
				[
					'label'      => __( 'Height', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ '%', 'px' ],
					'default'    => [
						'unit' => '%',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 100,
						],
						'%'  => [
							'min' => 1,
							'max' => 100,
						],
					],
					'condition'  => [
						'divider' => 'yes',
						'view'    => 'inline-items',
					],
					'selectors'  => [
						'{{WRAPPER}} .power_iconilist.inline-items .power_iconilist_item:not(:last-child)' => 'height: {{SIZE}}{{UNIT}}',
					],
				]
			);
			
			$this->add_control(
				'divider_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'condition' => [
						'divider' => 'yes',
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconilist.block .power_iconilist_item:not(:last-child)'        => 'border-bottom-color:{{VALUE}};',
						'{{WRAPPER}} .power_iconilist.inline-items .power_iconilist_item:not(:last-child)' => 'border-right-color:{{VALUE}};',
					],
				]
			);
			
			$this->end_controls_section();
			
			$this->start_controls_section(
				'section_style_icon',
				[
					'label' => __( 'Icon', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			
			$this->add_control(
				'icon_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item i' => 'color: {{VALUE}};',
					]
				]
			);
			
			$this->add_control(
				'icon_color_hover',
				[
					'label'     => __( 'Hover', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item:hover i, {{WRAPPER}} .power_iconilist .power_iconilist_item a:hover i' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->add_responsive_control(
				'icon_size',
				[
					'label'     => __( 'Size', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'default'   => [
						'size' => 14,
					],
					'range'     => [
						'px' => [
							'min' => 6,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item i, {{WRAPPER}} .power_iconilist .power_iconilist_item a i'   => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			
			$this->end_controls_section();
			
			$this->start_controls_section(
				'section_text_style',
				[
					'label' => __( 'Text', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			
			$this->add_control(
				'text_color',
				[
					'label'     => __( 'Text Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item a, {{WRAPPER}} .power_iconilist .power_iconilist_item a .power_iconilist_text, {{WRAPPER}} .power_iconilist .power_iconilist_item .power_iconilist_text' => 'color: {{VALUE}};',
					]
				]
			);
			
			$this->add_control(
				'text_color_hover',
				[
					'label'     => __( 'Hover', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item:hover a, {{WRAPPER}} .power_iconilist .power_iconilist_item a:hover .power_iconilist_text, {{WRAPPER}} .power_iconilist .power_iconilist_item:hover .power_iconilist_text' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'text_indent',
				[
					'label'     => __( 'Text Indent', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'max' => 50,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconilist .power_iconilist_item a, {{WRAPPER}} .power_iconilist .power_iconilist_item a .power_iconilist_text, {{WRAPPER}} .power_iconilist .power_iconilist_item .power_iconilist_text' => 'padding-left: {{SIZE}}{{UNIT}};',
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'icon_typography',
					'selector' => '{{WRAPPER}} .power_iconilist .power_iconilist_item',
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_3,
				]
			);
			
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			$html     = '';
			if ( $settings['icon_list'] ) {
				$this->add_render_attribute( 'list', 'class', [
					'power_iconilist',
					$settings['view'],
					$settings['align']
				] );
				$html = '<div class="power_iconlist_wrapper">
    			<ul ' . $this->get_render_attribute_string( 'list' ) . '>';
				foreach ( $settings['icon_list'] as $item ) {
					if ( ! empty( $item['link']['url'] ) ) {
						$this->add_render_attribute( 'button', 'href', $item['link']['url'] );
						if ( $item['link']['is_external'] ) {
							$this->add_render_attribute( 'button', 'target', '_blank' );
						}
						if ( $item['link']['nofollow'] ) {
							$this->add_render_attribute( 'button', 'rel', 'nofollow' );
						}
						
						$link_html_start = '<a ' . $this->get_render_attribute_string( 'button' ) . '>';
						$link_html_end   = '</a>';
					} else {
						$link_html_start = $link_html_end = '';
					}
					
					$this->add_render_attribute( 'icon', 'class', [] );
					
					if ( $item['icon_align'] == 'left' ) {
						$icon_text_html = '<span class="power_iconilist_icon ' . esc_attr( $item['icon_align'] ) . '"><i class="' . esc_attr( $item['icon'] ) . '"></i></span>
		    				<span class="power_iconilist_text">' . esc_html( $item['text'] ) . '</span>';
					} else {
						$icon_text_html = '<span class="power_iconilist_text">' . esc_html( $item['text'] ) . '</span><span class="power_iconilist_icon ' . esc_attr( $item['icon_align'] ) . '"><i class="' . esc_attr( $item['icon'] ) . '"></i></span>';
					}
					
					$html .= '<li class="power_iconilist_item">
		    			' . $link_html_start . '
		    			' . $icon_text_html . '
		    			' . $link_html_end . '
		    		</li>';
				}
				$html .= '</ul></div>';
			}
			echo $html;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#

            view.addRenderAttribute( 'list', {
            'class': [ 'power_iconilist', settings.view,  settings.align]
            } );

            #>

            <# if ( settings.icon_list ) { #>
            <div class="power_iconlist_wrapper">
                <ul {{{ view.getRenderAttributeString(
                'list' ) }}}>
                <# _.each( settings.icon_list, function( item, index ) { #>
                <li class="power_iconilist_item">
                    <# if ( item.link && item.link.url ) { #>
                    <a href="{{ item.link.url }}">
                        <# } #>
                        <# if ( item.icon_align=='left' ) { #>
                        <# if ( item.icon ) { #>
                        <span class="power_iconilist_icon {{item.icon_align}}">
							<i class="{{ item.icon }}" aria-hidden="true"></i>
						</span>
                        <# } #>
                        <span class="power_iconilist_text">{{{ item.text }}}</span>
                        <# }else{ #>
                        <span class="power_iconilist_text">{{{ item.text }}}</span>
                        <# if ( item.icon ) { #>
                        <span class="power_iconilist_icon {{item.icon_align}}">
							<i class="{{ item.icon }}" aria-hidden="true"></i>
						</span>
                        <# } #>
                        <# }#>
                        <# if ( item.link && item.link.url ) { #>
                    </a>
                    <# } #>
                </li>
                <#
                } ); #>
                </ul>
            </div>
            <#    } #>
			
			<?php
		}
	}