<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Icon_Box extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_icon_box';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Icon Box', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-icon-box';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-icon-box' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_list',
				[
					'label' => __( 'Icon List', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'icon_box_style',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'type_1',
					'options' => [
						'type_1' => __( 'Type 1', 'power_elements_elementor' ),
						'type_2' => __( 'Type 2', 'power_elements_elementor' )
					]
				]
			);
			$this->add_responsive_control(
				'align',
				[
					'label'     => __( 'Alignment', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::CHOOSE,
					'options'   => [
						'left'   => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center' => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'  => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						]
					],
					'default'   => 'center',
					'condition' => [
						'icon_box_style' => 'type_2',
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconbox.type_2 .power_iconbox_wrapper' => 'text-align: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'icon_image',
				[
					'label'        => __( 'Icon or Image', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SWITCHER,
					'label_on'     => __( 'Icon', 'power_elements_elementor' ),
					'label_off'    => __( 'Image', 'power_elements_elementor' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			
			$this->add_control(
				'icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
					'condition'   => [
						'icon_image' => 'yes',
					],
				]
			);
			$this->add_control(
				'box_image',
				[
					'label'     => __( 'Choose Image', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::MEDIA,
					'dynamic'   => [
						'active' => true,
					],
					'default'   => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
					'condition' => [
						'icon_image!' => 'yes',
					],
				]
			);
			$this->add_control(
				'box_number',
				[
					'label'       => __( 'Number', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( '01', 'power_elements_elementor' ),
					'placeholder' => __( 'Type your number here', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'box_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Default title', 'power_elements_elementor' ),
					'placeholder' => __( 'Type your title here', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'box_description',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'rows'        => 10,
					'default'     => __( 'Default description', 'power_elements_elementor' ),
					'placeholder' => __( 'Type your description here', 'power_elements_elementor' ),
				]
			);
			
			$this->end_controls_section();
			/*
			 * STYLE OPTIONS
			 */
			$this->start_controls_section(
				'section_style',
				[
					'label' => __( 'Box', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'icon_box_tabs'
			);
			
			$this->start_controls_tab(
				'icon_box_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'icon_box_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_iconbox' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'icon_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'icon_box_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox',
				]
			);
			$this->add_control(
				'icon_box_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * HOVER
			 */
			$this->start_controls_tab(
				'icon_box_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'icon_box_bg_color_hover',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_iconbox:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'icon_box_shadow_hover',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox:hover',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'icon_box_border_hover',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox:hover',
				]
			);
			$this->add_control(
				'icon_box_radius_hover',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_padding_hover',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox:hover' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_margin_hover',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox:hover' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * STYLE OPTIONS WRAPPER
			 */
			$this->start_controls_section(
				'section_wrapper_style',
				[
					'label' => __( 'Wrapper', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'icon_box_wrapper_tabs'
			);
			
			$this->start_controls_tab(
				'icon_box_wrapper_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'icon_box_wrapper_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'icon_box_wrapper_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox .power_iconbox_wrapper',
				]
			);
			$this->add_control(
				'icon_box_wrapper_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_wrapper_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_wrapper_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * HOVER
			 */
			$this->start_controls_tab(
				'icon_box_wrapper_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name' => 'icon_box_wrapper_bg_color_hover',
					'label' => __( 'Background', 'power_elements_elementor' ),
					'types' => [ 'gradient'],
					'selector' => '{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'icon_box_wrapper_border_hover',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper',
				]
			);
			$this->add_control(
				'icon_box_wrapper_radius_hover',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_wrapper_padding_hover',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'icon_box_wrapper_margin_hover',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * STYLE OPTIONS ICON
			 */
			$this->start_controls_section(
				'section_icon_style',
				[
					'label' => __( 'Icon', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'icon_box_icon_tabs'
			);
			$this->start_controls_tab(
				'icon_box_icon_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'icon_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_icon' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_responsive_control(
				'icon_size',
				[
					'label'     => __( 'Size', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'default'   => [
						'size' => 36,
					],
					'range'     => [
						'px' => [
							'min' => 6,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_icon'   => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * HOVER
			 */
			$this->start_controls_tab(
				'icon_box_icon_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'icon_hover_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_icon' => 'color: {{VALUE}};',
					]
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();

			/*
			 * STYLE OPTIONS NUMBER
			 */
			$this->start_controls_section(
				'section_number_style',
				[
					'label' => __( 'Number', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'icon_box_number_tabs'
			);
			$this->start_controls_tab(
				'icon_box_number_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'number_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h6' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'number_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h6',
				]
			);
			$this->end_controls_tab();
			/*
			 * HOVER
			 */
			$this->start_controls_tab(
				'icon_box_number_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'number_hover_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper .power_iconbox_content h6' => 'color: {{VALUE}};',
					]
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();

			/*
			 * STYLE OPTIONS TITLE
			 */
			$this->start_controls_section(
				'section_title_style',
				[
					'label' => __( 'Title', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'icon_box_title_tabs'
			);
			$this->start_controls_tab(
				'icon_box_title_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h3' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h3',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'title_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h3',
				]
			);
			$this->add_control(
				'title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h3' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content h3' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * HOVER
			 */
			$this->start_controls_tab(
				'icon_box_title_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'title_hover_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper .power_iconbox_content h3' => 'color: {{VALUE}};',
					]
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * STYLE OPTIONS DESCRIPTION
			 */
			$this->start_controls_section(
				'section_des_style',
				[
					'label' => __( 'Description', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'icon_box_des_tabs'
			);
			$this->start_controls_tab(
				'icon_box_des_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content p' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content p',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content p',
				]
			);
			$this->add_responsive_control(
				'des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * HOVER
			 */
			$this->start_controls_tab(
				'icon_box_des_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'des_hover_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_iconbox:hover .power_iconbox_wrapper .power_iconbox_content p' => 'color: {{VALUE}};',
					]
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			$this->start_controls_section(
				'section_gap_style',
				[
					'label' => __( 'Gap', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'gap_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_iconbox .power_iconbox_wrapper .power_iconbox_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( $settings['icon_image'] == 'yes' ) {
				$icon_image = '<div class="power_iconbox_icon">
                        <i class="' . esc_attr( $settings['icon'] ) . '"></i>
                    </div>';
			} else {
				$icon_image = '<div class="power_iconbox_icon">
                        <img src="' . esc_url( $settings['box_image']['url'] ) . '" alt="' . esc_attr( $settings['box_title'] ) . '">
                    </div>';
			}
			
			echo '<div class="power_iconbox ' . esc_attr( $settings['icon_box_style'] ) . '">
                <div class="power_iconbox_wrapper">
                   ' . $icon_image . '
                    <div class="power_iconbox_content">
                        <h6>' . esc_html( $settings['box_number'] ) . '</h6>
                        <h3>' . esc_html( $settings['box_title'] ) . '</h3>
                        <p>' . esc_html( $settings['box_description'] ) . '</p>
                    </div>
                </div>
            </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
			<#
			var icon_image ='';
			if ( settings.icon_image=='yes' ) {
				icon_image ='<div class="power_iconbox_icon"><i class="'+ settings.icon +'"></i></div>';
			}else{
				icon_image ='<div class="power_iconbox_icon"><img src=" '+ settings.box_image.url +' " alt="'+settings.box_title+'"/></div>';
			}
			#>
			<div class="power_iconbox {{settings.icon_box_style}}">
				<div class="power_iconbox_wrapper">
					{{{icon_image}}}
					<div class="power_iconbox_content">
						<h6>{{{settings.box_number}}}</h6>
						<h3>{{{settings.box_title}}}</h3>
						<p>{{{settings.box_description}}}</p>
					</div>
				</div>
			</div>
			<?php
		}
		
	}