<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Toggle extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_toggle';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Toggle', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-accordion';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-toggle' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-toggle-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_toggle_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'toggle_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Title', 'power_elements_elementor' ),
					'placeholder' => __( 'Give your first toggle title', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'toggle_content',
				[
					'label'       => __( 'Content', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::WYSIWYG,
					'default'     => __( 'Content', 'power_elements_elementor' ),
					'placeholder' => __( 'Give your toggle content', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'list',
				[
					'label'       => __( 'toggle', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'list_title'   => __( 'Title #1', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
						[
							'list_title'   => __( 'Title #2', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
					],
					'title_field' => '{{{ toggle_title }}}'
				]
			);
			$this->add_control(
				'toggle_icon',
				[
					'label'       => __( 'toggle Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => 'fa fa-plus',
				]
			);
			$this->add_control(
				'toggle_open_icon',
				[
					'label'       => __( 'Open Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => 'fa fa-minus',
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE toggle START
			 */
			$this->start_controls_section(
				'power_element_toggle_style',
				[
					'label' => __( 'toggle', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'toggle_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE TITLE
			 */
			$this->start_controls_section(
				'power_element_toggle_title_style',
				[
					'label' => __( 'Title', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'style_toggle_title_tabs'
			);
			/*
			 * TITLE ACTIVE
			 */
			$this->start_controls_tab(
				'style_toggle_title_active_tab',
				[
					'label' => __( 'Active', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'toggle_title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title',
				]
			);
			$this->add_control(
				'toggle_title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'toggle_title_bg_color',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'toggle_title_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'toggle_title_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title',
				]
			);
			$this->add_control(
				'toggle_title_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * TITLE NORMAL
			 */
			$this->start_controls_tab(
				'style_toggle_title_normal_tab',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'toggle_title_typography_normal',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_toggle .item .power_toggle_tab .power_toggle_title',
				]
			);
			$this->add_control(
				'toggle_title_color_normal',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'toggle_title_bg_color_normal',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'toggle_title_shadow_normal',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'toggle_title_border_normal',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title',
				]
			);
			$this->add_control(
				'toggle_title_radius_normal',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_title_margin_normal',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_title_padding_normal',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item:not(.active) .power_toggle_tab .power_toggle_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * TITLE HOVER
			 */
			$this->start_controls_tab(
				'style_toggle_title_hover_tab',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'toggle_title_color_hover',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item:not(.active):hover .power_toggle_tab .power_toggle_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'toggle_title_bg_color_hover',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item:not(.active):hover .power_toggle_tab .power_toggle_title' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'toggle_title_border_hover',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item::not(.active)hover .power_toggle_tab .power_toggle_title',
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * ICON STYLE
			 */
			$this->start_controls_section(
				'power_element_toggle_icon_style',
				[
					'label' => __( 'Icon', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'toggle_icon_align',
				[
					'label'       => __( 'Alignment', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::CHOOSE,
					'options'     => [
						'toggle-left'  => [
							'title' => __( 'Start', 'power_elements_elementor' ),
							'icon'  => 'eicon-h-align-left',
						],
						'toggle-right' => [
							'title' => __( 'End', 'power_elements_elementor' ),
							'icon'  => 'eicon-h-align-right',
						],
					],
					'default'     => 'toggle-right',
					'toggle'      => false,
					'label_block' => false
				]
			);
			$this->add_control(
				'toggle_icon_left',
				[
					'label'      => __( 'Left', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
						'%'  => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 10,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_tab .power_toggle_tab_icon.toggle-left' => 'left: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_icon_right',
				[
					'label'      => __( 'Right', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
						'%'  => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 10,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_tab .power_toggle_tab_icon.toggle-right' => 'right: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_icon_top',
				[
					'label'      => __( 'Top', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
						'%'  => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => '%',
						'size' => 50,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_tab .power_toggle_tab_icon' => 'top: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_icon_open',
				[
					'label'     => __( 'Open Icon', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_control(
				'toggle_icon_font_size',
				[
					'label'      => __( 'Font Size', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', 'em' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
						'em' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 24,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_tab_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_icon_font_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item.active .power_toggle_tab .power_toggle_tab_icon i' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'toggle_icon_close',
				[
					'label'     => __( 'Close Icon', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->start_controls_tabs(
				'style_toggle_close_icon_tabs'
			);
			/*
			 *
			 * ICON NORMAL
			 */
			$this->start_controls_tab(
				'style_toggle_close_icon_tabs_normal',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'toggle_close_icon_font_size',
				[
					'label'      => __( 'Font Size', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', 'em' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
						'em' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 20,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_tab .power_toggle_tab_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_close_icon_font_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_tab .power_toggle_tab_icon i' => 'color: {{VALUE}}',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 * ICON HOVER
			 */
			$this->start_controls_tab(
				'style_toggle_close_icon_tabs_hover',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'toggle_close_icon_font_size_hover',
				[
					'label'      => __( 'Font Size', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', 'em' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
						'em' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 20,
					],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item:hover .power_toggle_tab .power_toggle_tab_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_close_icon_font_color_hover',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item:hover .power_toggle_tab .power_toggle_tab_icon i' => 'color: {{VALUE}}',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
			/*
			 * CONTENT STYLE
			 */
			$this->start_controls_section(
				'power_element_toggle_content_style',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'toggle_content_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_toggle .item .power_toggle_info',
				]
			);
			$this->add_control(
				'toggle_content_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_info' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'toggle_content_bg_color',
				[
					'label'     => __( 'Background', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_info' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'toggle_content_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item .power_toggle_info',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'toggle_content_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_toggle .item .power_toggle_info',
				]
			);
			$this->add_control(
				'toggle_content_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_info' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_content_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_info' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'toggle_content_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_toggle .item .power_toggle_info' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			$html     = '';
			
			if ( $settings['list'] ) {
				$html = '<div class="power_toggle">';
				foreach ( $settings['list'] as $key => $item ) {
					if ( $key == 0 ) {
						$active = 'active';
					} else {
						$active = '';
					}
					$html .= '<div class="item ' . esc_attr( $active ) . '">
                    <div class="power_toggle_tab">
                        <div class="power_toggle_title ' . esc_attr( $settings['toggle_icon_align'] ) . '">' . esc_html( $item['toggle_title'] ) . '</div>
                        <span class="power_toggle_tab_icon ' . esc_attr( $settings['toggle_icon_align'] ) . '">
                            <i class="open_icon ' . esc_attr( $settings['toggle_icon'] ) . '"></i>
                            <i class="close_icon ' . esc_attr( $settings['toggle_open_icon'] ) . '"></i>
                        </span>
                    </div>
                    <div class="power_toggle_info ' . esc_attr( $active ) . '">
                         ' . $item['toggle_content'] . '
                    </div>
                </div>';
				}
				$html .= '</div>';
			}
			echo $html;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <div class="power_toggle">
                <#
                var active='';
                _.each( settings.list, function( item , i) {
                if(i==0){
                active = 'active';
                }else{
                active ='';
                }
                #>
                <div class="item {{active}}">
                    <div class="power_toggle_tab">
                        <div class="power_toggle_title {{settings.toggle_icon_align}}">
                            {{{item.toggle_title}}}
                        </div>
                        <span class="power_toggle_tab_icon {{settings.toggle_icon_align}}">
                            <i class="open_icon {{settings.toggle_icon}}"></i>
                            <i class="close_icon {{settings.toggle_open_icon}}"></i>
                        </span>
                    </div>
                    <div class="power_toggle_info {{active}}">
                        {{{item.toggle_content}}}
                    </div>
                </div>
                <# }); #>
            </div>
			<?php
		}
	}