<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Section extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_section';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Info Section', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-info-box';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-section' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-section-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_tab_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'info_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Default title', 'power_elements_elementor' ),
					'placeholder' => __( 'Type your title here', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'info_description',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'rows'        => 10,
					'default'     => __( 'Default description', 'power_elements_elementor' ),
					'placeholder' => __( 'Type your description here', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'info_style',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'type_1',
					'options' => [
						'type_1' => __( 'Type 1', 'power_elements_elementor' ),
						'type_2' => __( 'Type 2', 'power_elements_elementor' )
					]
				]
			);
			$this->add_responsive_control(
				'align',
				[
					'label'   => __( 'Alignment', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::CHOOSE,
					'options' => [
						'align-left'   => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'align-center' => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'align-right'  => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						]
					],
					'default' => 'align-left',
				]
			);
			$this->add_control(
				'button_show_hide',
				[
					'label'        => __( 'Button', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SWITCHER,
					'label_on'     => __( 'show', 'power_elements_elementor' ),
					'label_off'    => __( 'Hide', 'power_elements_elementor' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$this->add_control(
				'link',
				[
					'label'       => __( 'Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'default'     => [
						'url' => '#',
					],
					'condition'   => [
						'button_show_hide' => 'yes',
					],
				]
			);
			$this->add_control(
				'button_text',
				[
					'label'       => __( 'Button Text', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Click here', 'power_elements_elementor' ),
					'placeholder' => __( 'Click here', 'power_elements_elementor' ),
					'condition'   => [
						'button_show_hide' => 'yes',
					]
				]
			);
			$this->add_control(
				'button_icon',
				[
					'label'       => __( 'Button Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
					'condition'   => [
						'button_show_hide' => 'yes',
					]
				]
			);
			$this->add_control(
				'external_url',
				[
					'label'     => __( 'Video', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SWITCHER,
					'label_on'  => __( 'Self', 'power_elements_elementor' ),
					'label_off' => __( 'Hosted', 'power_elements_elementor' ),
					'condition' => [
						'info_style' => 'type_2',
					],
				]
			);
			$this->add_control(
				'self_url',
				[
					'label'       => __( 'Self Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'placeholder' => __( 'Enter your URL', 'power_elements_elementor' ),
					'default'     => 'https://www.youtube.com/watch?v=opj24KnzrWo',
					'label_block' => true,
					'condition'   => [
						'external_url' => 'yes',
						'info_style' => 'type_2',
					],
				]
			);
			$this->add_control(
				'hosted_url',
				[
					'label'      => __( 'Choose MP4 File', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::MEDIA,
					'media_type' => 'video',
					'condition'  => [
						'external_url' => '',
						'info_style' => 'type_2',
					],
				]
			);
			$this->add_control(
				'icon',
				[
					'label'       => __( 'Play Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => 'fa fa-play',
					'condition'   => [
						'info_style' => 'type_2',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE OPTIONS
			 */
			$this->start_controls_section(
				'section_info_style',
				[
					'label' => __( 'Info', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name' => 'info_background',
					'label' => __( 'Background', 'power_elements_elementor' ),
					'types' => [ 'classic', 'gradient', 'video' ],
					'selector' => '{{WRAPPER}} .power_header_content',
				]
			);
			$this->add_control(
				'info_padding',
				[
					'label' => __( 'Padding', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors' => [
						'{{WRAPPER}} .power_header_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * STYLE CONTENT WRAPPER
			 */
			$this->start_controls_section(
				'section_wrapper_style',
				[
					'label' => __( 'Content Wrapper', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'contentwrap_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_header_content .power_header_inner' => 'background-color: {{VALUE}}',
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'contentwrap_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_content .power_header_inner',
				]
			);
			$this->add_control(
				'contentwrap_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_content .power_header_inner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'contentwrap_padding',
				[
					'label' => __( 'Padding', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors' => [
						'{{WRAPPER}} .power_header_content .power_header_inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'contentwrap_margin',
				[
					'label' => __( 'Margin', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors' => [
						'{{WRAPPER}} .power_header_content .power_header_inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * STYLE CONTENT WRAPPER
			 */
			$this->start_controls_section(
				'section_inner_style',
				[
					'label' => __( 'Content Inner', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'content_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_header_content .power_header_top' => 'background-color: {{VALUE}}',
					],
				]
			);

			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'content_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_content .power_header_top',
				]
			);
			$this->add_control(
				'content_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_content .power_header_top' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'content_padding',
				[
					'label' => __( 'Padding', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors' => [
						'{{WRAPPER}} .power_header_content .power_header_top' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'content_margin',
				[
					'label' => __( 'Margin', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors' => [
						'{{WRAPPER}} .power_header_content .power_header_top' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE TITLE
			 */
			$this->start_controls_section(
				'section_title_style',
				[
					'label' => __( 'Title', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_content h3' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_content h3',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'title_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_content h3',
				]
			);
			$this->add_control(
				'title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_content h3' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_content h3' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE DESCRIPTION
			 */
			$this->start_controls_section(
				'section_des_style',
				[
					'label' => __( 'Description', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_content p' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_content p',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_content p',
				]
			);
			$this->add_control(
				'des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_content p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_content p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE BUTTON
			 */
			$this->start_controls_section(
				'section_button_style',
				[
					'label' => __( 'Button', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'button_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_button' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_hover_color',
				[
					'label'     => __( 'Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_button:hover' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'button_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_button',
				]
			);
			$this->add_control(
				'button_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_button:after' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_bg_hover_color',
				[
					'label'     => __( 'Hover Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_button:before' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_bg_hidden_color',
				[
					'label'     => __( 'Hidden Bg Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_header_button' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'button_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_button',
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'button_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_header_button',
				]
			);
			$this->add_control(
				'button_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'button_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_button_group' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'button_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_header_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$title       = $settings['info_title'];
			$description = $settings['info_description'];
			if ( $title ) {
				$title = '<h3 class="power_header_title">' . Power_elements_Helper::power_kses( $title ) . '</h3>';
			}
			if ( $description ) {
				$description = '<p class="power_header_desc">' . Power_elements_Helper::power_kses( $description ) . '</p>';
			}
			
			if ( $settings['info_style'] == 'type_2' ) {
				$video_html = $this->get_video_html();
			} else {
				$video_html = '';
			}
			
			if ( $settings['button_show_hide'] ) {
				if ( ! empty( $settings['link']['url'] ) ) {
					$this->add_render_attribute( 'button', 'href', $settings['link']['url'] );
					if ( $settings['link']['is_external'] ) {
						$this->add_render_attribute( 'button', 'target', '_blank' );
					}
					if ( $settings['link']['nofollow'] ) {
						$this->add_render_attribute( 'button', 'rel', 'nofollow' );
					}
				}
				
				$this->add_render_attribute( 'button', 'class', [
					'power_header_button',
					'effect_5',
					'power_header_button_icon_right',
				] );
				$this->add_render_attribute( 'button', 'role', 'button' );
				if ( $settings['button_icon'] ) {
					$button_icon = '<i class="' . esc_attr( $settings['button_icon'] ) . ' power_header_button_icon"></i>';
				} else {
					$button_icon = '';
				}
				$button_html = '<div class="power_header_button_group" id="power_header_button_id">
                            <a ' . $this->get_render_attribute_string( 'button' ) . ' ><!--effect_(1-5)-->
                                <span class="button_value">' . Power_elements_Helper::power_kses( $settings['button_text'] ) . $button_icon . ' </span>
                            </a>
                        </div>';
			} else {
				$button_html = '';
			}
			
			echo '<div class="power_header_content ' . esc_attr( $settings['info_style'] ) . '">
                <div class="power_header_inner ' . esc_attr( $settings['align'] ) . '">
                    <div class="power_header_top">
                        ' . $title . $description . '
                        ' . $button_html . '
                    </div>
                </div>
                ' . $video_html . '
            </div>';
		}
		
		/*
		 * Button HTML
		 */
		function get_video_html() {
			$settings = $this->get_settings_for_display();
			
			if ( 'yes' === $settings['external_url'] ) {
				$video_url = $settings['self_url'];
			} else {
				$video_url = $settings['hosted_url']['url'];
			}
			
			return '<div class="power_header_video">
                    <span class="power_header_video_button">
                        <a class="fancybox-media" href="' . esc_url( $video_url ) . '"><i class="' . esc_attr( $settings['icon'] ) . '"></i></a>
                    </span>
                </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		
	}