<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Funfact extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_funfact';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Fun Fact', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-nerd-chuckle';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-funfact' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-funfact-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_funfact_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'funfact_type',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'type1',
					'options' => [
						'type1' => __( 'Type 1', 'power_elements_elementor' ),
						'type2' => __( 'Type 2', 'power_elements_elementor' ),
						'type3' => __( 'Type 3', 'power_elements_elementor' )
					],
				]
			);
			$this->add_control(
				'funfact_number',
				[
					'label'   => __( 'Number', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::NUMBER,
					'min'     => 1,
					'max'     => 10000,
					'step'    => 1,
					'default' => 75,
				]
			);
			$this->add_control(
				'funfact_suffix',
				[
					'label'   => __( 'Suffix', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => '',
					'options' => [
						'+' => __( 'Yes', 'power_elements_elementor' ),
						'' => __( 'No', 'power_elements_elementor' )
					],
				]
			);
			$this->add_control(
				'funfact_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Default title', 'power_elements_elementor' ),
					'placeholder' => __( 'Type your title here', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'funfact_desc',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( '', 'power_elements_elementor' ),
					'placeholder' => __( 'Short description here', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'funfact_switcher',
				[
					'label' => __( 'Image', 'power_elements_elementor' ),
					'type'  => \Elementor\Controls_Manager::SWITCHER,
				]
			);
			$this->add_control(
				'icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
					'condition'   => [
						'funfact_switcher' => '',
					],
				]
			);
			$this->add_control(
				'image',
				[
					'label'     => __( 'Choose Image', 'plugin-domain' ),
					'type'      => \Elementor\Controls_Manager::MEDIA,
					'default'   => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
					'condition' => [
						'funfact_switcher' => 'yes',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_style',
				[
					'label'     => __( 'Funfact', 'power_elements_elementor' ),
					'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
					'condition' => [
						'funfact_type' => 'type1',
					],
				]
			);
			$this->add_responsive_control(
				'space_between',
				[
					'label'     => __( 'Space Between', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_funbox.type1 .power_fun_content' => 'padding-left: calc({{SIZE}}{{UNIT}}/2);'
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'funfact_number_style',
				[
					'label'     => __( 'Number', 'power_elements_elementor' ),
					'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'number_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_funbox .power_fun_value',
				]
			);
			$this->add_control(
				'number_text_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_funbox .power_fun_value' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'number_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_funbox .power_fun_value',
				]
			);
			$this->add_control(
				'number_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_funbox .power_fun_value' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'number_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_funbox .power_fun_value' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * TITLE STYLE
			 */
			$this->start_controls_section(
				'funfact_title_style',
				[
					'label'     => __( 'Title', 'power_elements_elementor' ),
					'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_funbox .power_fun_title',
				]
			);
			$this->add_control(
				'title_text_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_funbox .power_fun_title' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'title_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_funbox .power_fun_title',
				]
			);
			$this->add_control(
				'title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_funbox .power_fun_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_funbox .power_fun_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		

		/*
			 * Description STYLE
			 */
			$this->start_controls_section(
				'funfact_desc_style',
				[
					'label'     => __( 'Description', 'power_elements_elementor' ),
					'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'desc_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_funbox .power_fun_desc',
				]
			);
			$this->add_control(
				'desc_text_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_funbox .power_fun_desc' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'desc_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_funbox .power_fun_desc' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'desc_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_funbox .power_fun_desc' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( $settings['funfact_switcher'] ) {
				$html = ' <div class="power_fun_image">
                    <img src="' . esc_url( $settings['image']['url'] ) . '" alt="' . Power_elements_Helper::power_kses( $settings['funfact_title'] ) . '">
                </div>';
			} else {
				$html = ' <div class="power_fun_image">
                    <i class="' . esc_attr( $settings['icon'] ) . '"></i>
                </div>';
			}
			echo '<div id="power_funbox" class="power_funbox ' . esc_attr( $settings['funfact_type'] ) . '">
               ' . $html . '
                <div class="power_fun_content">
                    <h1 class="power_fun_value"><span class="power_fun_number">' . Power_elements_Helper::power_kses( $settings['funfact_number'] ) . '</span>' . Power_elements_Helper::power_kses( $settings['funfact_suffix'] ) . '</h1>
                    <h3 class="power_fun_title">' . Power_elements_Helper::power_kses( $settings['funfact_title'] ) . '</h3>
                    <p class="power_fun_desc">' . Power_elements_Helper::power_kses( $settings['funfact_desc'] ) . '</p>
                </div>
            </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <div id="power_funbox" class="power_funbox {{settings.funfact_type}}">
                <# if(settings.funfact_switcher){#>
                <div class="power_fun_image">
                    <img src="{{settings.image.url}}" alt="{{settings.funfact_title}}">
                </div>
                <# } else{ #>
                <div class="power_fun_image">
                    <i class="{{settings.icon}}"></i>
                </div>
                <# } #>
                <div class="power_fun_content">
                    <h1 class="power_fun_value"><span class="power_fun_number">{{{settings.funfact_number}}}</span>{{{settings.funfact_suffix}}}</h1>
                    <h3 class="power_fun_title">{{{settings.funfact_title}}}</h3>
                    <p class="power_fun_desc">{{{settings.funfact_desc}}}</p>
                </div>
            </div>
			<?php
		}
	}