<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element button widget.
	 *
	 * Elementor widget that displays a button with the ability to control every
	 * aspect of the button design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Button extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve button widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_button';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve button widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Button', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve button widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-button';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the button widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-button' ];
		}
		
		/**
		 * Register button widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_button',
				[
					'label' => __( 'Button', 'power_elements_elementor' ),
				]
			);
			
			$this->add_control(
				'button_style',
				[
					'label'        => __( 'Type', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SELECT,
					'default'      => 'effect_1',
					'options'      => [
						'effect_1'  => __( 'Effect 1', 'power_elements_elementor' ),
						'effect_2'  => __( 'Effect 2', 'power_elements_elementor' ),
						'effect_3'  => __( 'Effect 3', 'power_elements_elementor' ),
						'effect_4'  => __( 'Effect 4', 'power_elements_elementor' ),
						'effect_5'  => __( 'Effect 5', 'power_elements_elementor' ),
						'effect_6'  => __( 'Effect 6', 'power_elements_elementor' ),
						'effect_7'  => __( 'Effect 7', 'power_elements_elementor' ),
						'effect_8'  => __( 'Effect 8', 'power_elements_elementor' ),
						'effect_9'  => __( 'Effect 9', 'power_elements_elementor' ),
						'effect_10' => __( 'Effect 10', 'power_elements_elementor' ),
						'effect_11' => __( 'Effect 11', 'power_elements_elementor' ),
						'effect_12' => __( 'Effect 12', 'power_elements_elementor' ),
						'effect_13' => __( 'Effect 13', 'power_elements_elementor' ),
					]
				]
			);
			$this->start_controls_tabs( 'tabs_button_text' );
			$this->start_controls_tab(
				'tab_button_normal',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'text',
				[
					'label'       => __( 'Text', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Click here', 'power_elements_elementor' ),
					'placeholder' => __( 'Click here', 'power_elements_elementor' ),
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'tab_button_before',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'text_before',
				[
					'label'       => __( 'Text before', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Click here', 'power_elements_elementor' ),
					'placeholder' => __( 'Click here', 'power_elements_elementor' ),
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'link',
				[
					'label'       => __( 'Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'default'     => [
						'url' => '#',
					],
				]
			);
			
			$this->add_responsive_control(
				'align',
				[
					'label'        => __( 'Alignment', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'prefix_class' => 'elementor%s-align-',
					'default'      => '',
				]
			);
			
			$this->add_control(
				'size',
				[
					'label'          => __( 'Size', 'power_elements_elementor' ),
					'type'           => \Elementor\Controls_Manager::SELECT,
					'default'        => 'sm',
					'options'        => self::get_button_sizes(),
					'style_transfer' => true,
				]
			);
			
			$this->add_control(
				'icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
				]
			);
			$this->add_control(
				'icon_align',
				[
					'label'     => __( 'Icon Position', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SELECT,
					'default'   => 'power_button_icon_left',
					'options'   => [
						'power_button_icon_right' => __( 'Right', 'power_elements_elementor' ),
						'power_button_icon_left'  => __( 'Left', 'power_elements_elementor' ),
					],
					'condition' => [
						'icon!' => '',
					],
				]
			);
			
			$this->add_control(
				'icon_indent',
				[
					'label'     => __( 'Icon Spacing', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'max' => 50,
						],
					],
					'condition' => [
						'icon!' => '',
					],
					'selectors' => [
						'{{WRAPPER}} .power_button_icon_right i' => 'margin-left: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .power_button_icon_left i'  => 'margin-right: {{SIZE}}{{UNIT}};',
					],
				]
			);
			
			$this->add_control(
				'view',
				[
					'label'   => __( 'View', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::HIDDEN,
					'default' => 'traditional',
				]
			);
			$this->add_control(
				'button_css_id',
				[
					'label'       => __( 'Button ID', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => 'power-element-id',
					'title'       => __( 'Add your custom id WITHOUT the Pound key. e.g: my-id', 'power_elements_elementor' ),
					'label_block' => false,
					'description' => __( 'Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows <code>A-z 0-9</code> & underscore chars without spaces.', 'power_elements_elementor' ),
					'separator'   => 'before',
				
				]
			);
			
			$this->end_controls_section();
			$this->start_controls_section(
				'section_style',
				[
					'label' => __( 'Button', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'content_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Core\Schemes\Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_button',
				]
			);
			$this->start_controls_tabs( 'tabs_button_style' );
			
			$this->start_controls_tab(
				'tab_style_button_normal',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			
			$this->add_control(
				'button_text_color',
				[
					'label'     => __( 'Text Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .button_value, {{WRAPPER}} .button_value' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'background_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'scheme'    => [
						'type'  => \Elementor\Core\Schemes\Color::get_type(),
						'value' => \Elementor\Core\Schemes\Color::COLOR_4,
					],					
					'selectors' => [
						'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button' => 'background-color: {{VALUE}};',
					],
				]
			);
			

			$this->add_control(
				'button_background_after_color_label',
				[
					'label' => __( 'Background After Color', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'background_color_after',
					'label'    => __( 'Background After Color', 'power_elements_elementor' ),					
					'selector' => '{{WRAPPER}} a.power_button:after, {{WRAPPER}} .power_button:after',
				]
			);
			$this->end_controls_tab();
			
			$this->start_controls_tab(
				'tab_style_button_before',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			
			$this->add_control(
				'hover_text_color',
				[
					'label'     => __( 'Text Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_button:hover .button_value, {{WRAPPER}} .power_button:hover .button_value, {{WRAPPER}} a.power_button:focus, {{WRAPPER}} .power_button:focus' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'button_background_hover_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,	
					'selectors' => [
						'{{WRAPPER}} a.power_button:hover, {{WRAPPER}} .power_button:hover, {{WRAPPER}} a.power_button:focus, {{WRAPPER}} .power_button:focus' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'button_background_before_color_label',
				[
					'label' => __( 'Background Before Color', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'button_background_before_color',
					'label'    => __( 'Background Before Color', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} a.power_button:before, {{WRAPPER}} .power_button:before',
				]
			);

			$this->add_control(
				'button_hover_border_color',
				[
					'label'     => __( 'Border Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'condition' => [
						'border_border!' => '',
					],
					'selectors' => [
						'{{WRAPPER}} a.power_button:hover, {{WRAPPER}} .power_button:hover, {{WRAPPER}} a.power_button:focus, {{WRAPPER}} .power_button:focus' => 'border-color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'before_animation',
				[
					'label' => __( 'Hover Animation', 'power_elements_elementor' ),
					'type'  => \Elementor\Controls_Manager::HOVER_ANIMATION,
				]
			);
			
			$this->end_controls_tab();
			
			$this->end_controls_tabs();
			$this->add_control(
				'hr-i',
				[
					'type' => \Elementor\Controls_Manager::DIVIDER,
				]
			);
			$this->start_controls_tabs( 'tabs_button_border' );
			
			$this->start_controls_tab(
				'tab_border_button_normal',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			
			
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'      => 'border',
					'selector'  => '{{WRAPPER}} .power_button',
					'separator' => 'before',
				]
			);
			
			$this->add_control(
				'border_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'button_box_shadow',
					'selector' => '{{WRAPPER}} .power_button',
				]
			);
			
			$this->add_responsive_control(
				'text_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'tab_border_button_before',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'      => 'border_before',
					'selector'  => '{{WRAPPER}} .power_button:hover',
				]
			);
			
			$this->add_control(
				'before_border_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'before_button_box_shadow',
					'selector' => '{{WRAPPER}} .power_button:before',
				]
			);
			
			$this->add_responsive_control(
				'before_text_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_button:before, {{WRAPPER}} a.power_button:before, {{WRAPPER}} .power_button:before, {{WRAPPER}} .power_button:before' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_tab();
			$this->end_controls_section();
		}
		
		/**
		 * Get button sizes.
		 *
		 * Retrieve an array of button sizes for the button widget.
		 *
		 * @since 1.0.0
		 * @access public
		 * @static
		 *
		 * @return array An array containing button sizes.
		 */
		public static function get_button_sizes() {
			return [
				'xs' => __( 'Extra Small', 'power_elements_elementor' ),
				'sm' => __( 'Small', 'power_elements_elementor' ),
				'md' => __( 'Medium', 'power_elements_elementor' ),
				'lg' => __( 'Large', 'power_elements_elementor' ),
				'xl' => __( 'Extra Large', 'power_elements_elementor' ),
			];
		}
		
		/**
		 * Render button widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$this->add_render_attribute( 'wrapper', 'class', 'power_button-wrapper' );
			
			if ( ! empty( $settings['link']['url'] ) ) {
				$this->add_render_attribute( 'button', 'href', $settings['link']['url'] );
				$this->add_render_attribute( 'button', 'class', 'power_button-link' );
				
				if ( $settings['link']['is_external'] ) {
					$this->add_render_attribute( 'button', 'target', '_blank' );
				}
				
				if ( $settings['link']['nofollow'] ) {
					$this->add_render_attribute( 'button', 'rel', 'nofollow' );
				}
			}
			
			$this->add_render_attribute( 'button', 'class', [
				'power_button',
				$settings['button_style'],
				$settings['icon_align']
			] );
			$this->add_render_attribute( 'button', 'role', 'button' );
			$this->add_render_attribute( 'button', 'data-text', $settings['text_before'] );
			
			if ( ! empty( $settings['button_css_id'] ) ) {
				$this->add_render_attribute( 'button', 'id', $settings['button_css_id'] );
			}
			
			if ( ! empty( $settings['size'] ) ) {
				$this->add_render_attribute( 'button', 'class', 'elementor-size-' . $settings['size'] );
			}
			
			if ( $settings['before_animation'] ) {
				$this->add_render_attribute( 'button', 'class', 'elementor-animation-' . $settings['before_animation'] );
			}
			
			?>
            <div <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
                <a <?php echo $this->get_render_attribute_string( 'button' ); ?>>
					<?php $this->render_text(); ?>
                </a>
            </div>
			<?php
		}
		
		/**
		 * Render button text.
		 *
		 * Render button widget text.
		 *
		 * @since 1.5.0
		 * @access protected
		 */
		protected function render_text() {
			$settings = $this->get_settings_for_display();
			
			$this->add_render_attribute( [
				'content-wrapper' => [
					'class' => 'power_button-content-wrapper',
				],
				'text'            => [
					'class' => [ 'power_button-text', 'button_value' ]
				],
			] );
			
			$this->add_inline_editing_attributes( 'text', 'none' );
			?>
			<?php if ( $settings['icon_align'] == 'power_button_icon_right' ) : ?>
				<?php if ( ! empty( $settings['icon'] ) ) : ?>
                    <span <?php echo $this->get_render_attribute_string( 'text' ); ?>><?php echo $settings['text']; ?><i
                                class="<?php echo esc_attr( $settings['icon'] ); ?> power_button_icon"></i></span>
				<?php endif; ?>
			<?php else : ?>
                <span <?php echo $this->get_render_attribute_string( 'text' ); ?>><i
                            class="<?php echo esc_attr( $settings['icon'] ); ?> power_button_icon"></i><?php echo $settings['text']; ?></span>
			<?php endif; ?>
			<?php
		}
		
		
		/**
		 * Render button widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#
            if ( settings.link && settings.link.url ) {
            var link = settings.link.url
            }else{
            var link ='#';
            }
            view.addRenderAttribute( 'power-button', {
            'class': [ 'power_button','power_button-link', 'elementor-size-' + settings.size, settings.button_style,  settings.icon_align],
            'role': 'button',
            'href': link,
            'data-text': settings.text_before
            } );

            #>
            <div class="elementor-button-wrapper">
                <div class="power_button_group" id="{{ settings.button_css_id }}">
                    <a {{{ view.getRenderAttributeString( 'power-button' ) }}}>

                    <# if ( settings.icon_align=='power_button_icon_right' ) { #>
                    <span class="power_button-text  button_value">{{{ settings.text }}}<i
                                class="{{ settings.icon }} power_button_icon"></i></span>
                    <# }else{ #>
                    <span class="power_button-text  button_value"><i
                                class="{{ settings.icon }} power_button_icon"></i>{{{ settings.text }}}</span>
                    <# } #>
                    </a>
                </div>
            </div>
			<?php
		}
	}