<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Team extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_team';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Team', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-lock-user';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-team' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_team_content',
				[
					'label' => __( 'Team Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'team_style',
				[
					'label'   => __( 'Style', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'style_1',
					'options' => [
						'style_1' => __( 'Style 1', 'power_elements_elementor' ),
						'style_2' => __( 'Style 2', 'power_elements_elementor' ),
						'style_3' => __( 'Style 3', 'power_elements_elementor' )
					]
				]
			);
			$this->add_control(
				'name_text',
				[
					'label'       => __( 'Name', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Your Name', 'power_elements_elementor' ),
					'placeholder' => __( 'Your Name Please', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'designation_text',
				[
					'label'       => __( 'Address or Designation', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'CEO, Advertising Corp', 'power_elements_elementor' ),
					'placeholder' => __( 'Give your designation please.', 'power_elements_elementor' ),
					'label_block' => true
				]
			);
			$this->add_control(
				'description_text',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'default'     => __( '', 'power_elements_elementor' ),
					'placeholder' => __( 'Short description.', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'team_image',
				[
					'label'   => __( 'Choose Image', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			$this->add_control(
				'team_social_options',
				[
					'label'   => __( 'Social Link', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'hide',
					'options' => [
						'show' => __( 'Show', 'power_elements_elementor' ),
						'hide' => __( 'Hide', 'power_elements_elementor' )
					]
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'social_icon',
				[
					'label'   => __( 'Social Icons', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::ICON,
					'include' => [
						'fa fa-facebook',
						'fa fa-flickr',
						'fa fa-google-plus',
						'fa fa-instagram',
						'fa fa-linkedin',
						'fa fa-pinterest',
						'fa fa-reddit',
						'fa fa-twitch',
						'fa fa-twitter',
						'fa fa-vimeo',
						'fa fa-youtube',
					],
					'default' => 'fa fa-facebook',
				]
			);
			$repeater->add_control(
				'social_link',
				[
					'label'         => __( 'Link', 'power_elements_elementor' ),
					'type'          => \Elementor\Controls_Manager::URL,
					'placeholder'   => __( 'https://your-link.com', 'power_elements_elementor' ),
					'show_external' => true,
					'default'       => [
						'url'         => '',
						'is_external' => true,
						'nofollow'    => true,
					]
				]
			);
			$this->add_control(
				'list',
				[
					'label'       => __( 'Social List', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'list_title'   => __( 'Title #1', 'power_elements_elementor' ),
							'list_content' => __( 'Item content. Click the edit button to change this text.', 'power_elements_elementor' ),
						],
					],
					'title_field' => '{{{ social_icon }}}',
					'condition'   => [
						'team_social_options' => 'show',
					]
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'power_element_style',
				[
					'label' => __( 'Image Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'team_img_background',
					'label'    => __( 'Background', 'power_elements_elementor' ),
					'types'    => [ 'classic', 'gradient', 'video' ],
					'selector' => '{{WRAPPER}} .power_team_member_image',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'team_img_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_image',
				]
			);
			$this->add_control(
				'team_img_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'team_img_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'power_element_intro_style',
				[
					'label' => __( 'Intro Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
					'condition' => [
						'team_style' => ['style_2', 'style_3'],
					],
				]
			);
			$this->add_responsive_control(
				'team_intro_align',
				[
					'label'        => __( 'Alignment', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_intro' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'team_intro_background',
					'label'    => __( 'Background', 'power_elements_elementor' ),
					'types'    => [ 'classic', 'gradient'],
					'show_label'=>true,
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_intro',
				]
			);
			$this->add_control(
				'team_intro_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_intro' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'team_intro_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_intro' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			/*
			* Intro Name Style
			*/
			$this->add_control(
				'intro_name',
				[
					'label' => __( 'Intro Name', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_intro_title',
				]
			);
			$this->add_control(
				'name_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_intro_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'name_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_intro_title',
				]
			);
			$this->add_control(
				'name_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_intro_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'name_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_intro_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			/*
			* Intro Designation Style
			*/
			$this->add_control(
				'intro_designation',
				[
					'label' => __( 'Intro Designation', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'designation_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_intro_designation',
				]
			);
			$this->add_control(
				'designation_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_intro_designation' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'designation_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_intro_designation',
				]
			);
			$this->add_control(
				'designation_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_intro_designation' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'designation_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_intro_designation' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			/*
			* Intro Description Style
			*/
			$this->add_control(
				'intro_description',
				[
					'label' => __( 'Intro Description', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_text' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->end_controls_section();

			$this->start_controls_section(
				'power_element_details_style',
				[
					'label' => __( 'Details Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);

			$this->add_responsive_control(
				'team_details_align',
				[
					'label'        => __( 'Alignment', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'team_details_background',
					'label'    => __( 'Background', 'power_elements_elementor' ),
					'types'    => [ 'classic', 'gradient'],
					'show_label'=>true,
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_details',
				]
			);
			$this->add_control(
				'team_details_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'team_details_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_details' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);


			/*
			* Details Name Style
			*/
			$this->add_control(
				'detail_name',
				[
					'label' => __( 'Details Name', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'detail_title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_title',
				]
			);
			$this->add_control(
				'detail_name_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'detail_name_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_title',
				]
			);
			$this->add_control(
				'detail_name_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'detail_name_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			/*
			* Details Designation Style
			*/
			$this->add_control(
				'detail_designation',
				[
					'label' => __( 'Details Designation', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'detail_designation_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_designation',
				]
			);
			$this->add_control(
				'detail_designation_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_designation' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'detail_designation_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_designation',
				]
			);
			$this->add_control(
				'detail_designation_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_designation' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'detail_designation_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_designation' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			/*
			* Details Description Style
			*/
			$this->add_control(
				'detail_description',
				[
					'label' => __( 'Details Description', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'detail_des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'detail_des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member_text' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'detail_des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member_text',
				]
			);
			$this->add_control(
				'detail_des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'detail_des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->end_controls_section();
			
			
			/*
			 * Social Style
			 */
			$this->start_controls_section(
				'power_element_social_style',
				[
					'label' => __( 'Social Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'team_social_align',
				[
					'label'        => __( 'Alignment', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::CHOOSE,
					'options'      => [
						'flex-start'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'flex-end'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						]
					],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social' => 'align-items: {{VALUE}}; justify-content:{{VALUE}};',
					],
				]
			);
			$this->start_controls_tabs(
				'style_social_tabs'
			);
			/*
			 *
			 * SOCIAL NORMAL
			 */
			$this->start_controls_tab(
				'style_social_normal_tab',
				[
					'label' => __( 'Social Normal', 'plugin-name' ),
				]
			);
			$this->add_control(
				'social_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'social_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'social_width',
				[
					'label'     => __( 'Icon Width', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 40,
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'social_height',
				[
					'label'     => __( 'Icon Height', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 40,
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'social_lineheight',
				[
					'label'     => __( 'Line Height', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => 'px',
						'size' => 40,
					],
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'line-height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'team_social_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a',
				]
			);
			$this->add_control(
				'team_social_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			/*
			 *
			 * SOCIAL HOVER
			 */
			$this->start_controls_tab(
				'style_social_hover_tab',
				[
					'label' => __( 'Social Hover', 'plugin-name' ),
				]
			);
			$this->add_control(
				'social_hover_color',
				[
					'label'     => __( 'Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'social_bg_hover_color',
				[
					'label'     => __( 'Background Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'team_social_hover_border',
					'label'    => __( 'Hover Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover',
				]
			);
			$this->add_control(
				'team_social_hover_radius',
				[
					'label'      => __( 'Border Hover Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social .power_team_member_social_item a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'hr-tabs-end',
				[
					'type' => \Elementor\Controls_Manager::DIVIDER,
				]
			);
			$this->add_control(
				'social_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'social_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_team_member .power_team_member_social' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( $settings['team_image']['id'] != '' ) {
				$img = '<div class="power_team_member_image">
                            <img src="' . esc_url( $settings['team_image']['url'] ) . '" alt="' . esc_attr( $settings['name_text'] ) . '">
                        </div>';
			} else {
				$img = '';
			}
			
			if ( $settings['team_social_options'] == 'show' ) {
				$social = '<ul class="power_team_member_social">';
				foreach ( $settings['list'] as $item ) {
					$social .= '<li class="power_team_member_social_item">
                                <a href="' . esc_url( $item['social_link']['url'] ) . '"><i class="' . esc_attr( $item['social_icon'] ) . '"></i></a>
                            </li>';
				}
				$social .= '</ul>';
			} else {
				$social = '';
			}
			
			if ( $settings['description_text'] != '' ) {
				$description = '<p class="power_team_member_text">' . Power_elements_Helper::power_kses( $settings['description_text'] ) . '</p>';
			} else {
				$description = '';
			}
			
			if ( $settings['team_style'] == 'style_2' ) {
				echo '<div id="power_team_member_2" class="power_team_member overlay_style">
                 ' . $img . '
                <div class="power_team_member_intro">
	                 <h6 class="power_team_member_intro_designation">' . Power_elements_Helper::power_kses( $settings['designation_text'] ) . '</h5>
                     <h4 class="power_team_member_intro_title">' . Power_elements_Helper::power_kses( $settings['name_text'] ) . '</h3>
                </div>
                <div class="power_team_member_details">
                    <div class="power_team_member_inner">
                        ' . $social . '
						<h3 class="power_team_member_title">' . Power_elements_Helper::power_kses( $settings['name_text'] ) . '</h3>
	                     <h5 class="power_team_member_designation">' . Power_elements_Helper::power_kses( $settings['designation_text'] ) . '</h5>
                        ' . $description . '
                    </div>
                </div>
            </div>';
			} elseif ( $settings['team_style'] == 'style_3' ) {
				echo '<div id="power_team_member_3" class="power_team_member overlay_style">
               ' . $img . '
                <div class="power_team_member_details">
                    <div class="power_team_member_inner">
                        ' . $social . '
						<h3 class="power_team_member_title">' . Power_elements_Helper::power_kses( $settings['name_text'] ) . '</h3>
	                     <h5 class="power_team_member_designation">' . Power_elements_Helper::power_kses( $settings['designation_text'] ) . '</h5>
                        ' . $description . '
                    </div>
                </div>
            </div>';
			} else {
				echo '<div id="power_team_member_1" class="power_team_member simple_style">
                ' . $img . '
	                <div class="power_team_member_details">
	                    <div class="power_team_member_inner">
	                        <h5 class="power_team_member_designation">' . Power_elements_Helper::power_kses( $settings['designation_text'] ) . '</h5>
	                        <h3 class="power_team_member_title">' . Power_elements_Helper::power_kses( $settings['name_text'] ) . '</h3>
	                        ' . $description . $social . '
	                    </div>
	                </div>
            	</div>';
			}
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
			<# if ( settings.team_style == 'style_2' ) { #>
			<div id="power_team_member_2" class="power_team_member overlay_style">
				<# if ( settings.team_image.id != '' ) { #>
				<div class="power_team_member_image">
					<img src="{{settings.team_image.url}}" alt="{{settings.name_text}}">
				</div>
				<# } #>
				<div class="power_team_member_intro">
					<h6 class="power_team_member_intro_designation">{{{settings.designation_text}}}</h6>
					<h4 class="power_team_member_intro_title">{{{settings.name_text}}}</h4>
				</div>
				<div class="power_team_member_details">
					<div class="power_team_member_inner">
						<# if ( settings.team_social_options == 'show' ) { #>
						<ul class="power_team_member_social">
							<# _.each( settings.list, function( item ) { #>
							<li class="power_team_member_social_item">
								<a href="{{item.social_link.url}}"><i class="{{item.social_icon}}"></i></a>
							</li>
							<# }); #>
						</ul>
						<# } #>
						<h3 class="power_team_member_title">{{{settings.name_text}}}</h3>
						<h5 class="power_team_member_designation">{{{settings.designation_text}}}</h5>
						<# if ( settings.description_text != '' ) { #>
						<p class="power_team_member_text">{{{settings.description_text}}}</p>
						<# } #>
					</div>
				</div>
			</div>
			<# } else if(settings.team_style == 'style_3'){ #>
			<div id="power_team_member_3" class="power_team_member overlay_style">
				<# if ( settings.team_image.id != '' ) { #>
				<div class="power_team_member_image">
					<img src="{{settings.team_image.url}}" alt="{{settings.name_text}}">
				</div>
				<# } #>
				<div class="power_team_member_details">
					<div class="power_team_member_inner">
						<# if ( settings.team_social_options == 'show' ) { #>
						<ul class="power_team_member_social">
							<# _.each( settings.list, function( item ) { #>
							<li class="power_team_member_social_item">
								<a href="{{item.social_link.url}}"><i class="{{item.social_icon}}"></i></a>
							</li>
							<# }); #>
						</ul>
						<# } #>
						<h3 class="power_team_member_title">{{{settings.name_text}}}</h3>
						<h5 class="power_team_member_designation">{{{settings.designation_text}}}</h5>
						<# if ( settings.description_text != '' ) { #>
						<p class="power_team_member_text">{{{settings.description_text}}}</p>
						<# } #>
					</div>
				</div>
			</div>
			<# }else{#>
			<div id="power_team_member_1" class="power_team_member simple_style">
				<# if ( settings.team_image.id != '' ) { #>
				<div class="power_team_member_image">
					<img src="{{settings.team_image.url}}" alt="{{settings.name_text}}">
				</div>
				<# } #>
				<div class="power_team_member_details">
					<div class="power_team_member_inner">
						<h3 class="power_team_member_title">{{{settings.name_text}}}</h3>
						<h5 class="power_team_member_designation">{{{settings.designation_text}}}</h5>
						<# if ( settings.description_text != '' ) { #>
						<p class="power_team_member_text">{{{settings.description_text}}}</p>
						<# } #>
						<# if ( settings.team_social_options == 'show' ) { #>
						<ul class="power_team_member_social">
							<# _.each( settings.list, function( item ) { #>
							<li class="power_team_member_social_item">
								<a href="{{item.social_link.url}}"><i class="{{item.social_icon}}"></i></a>
							</li>
							<# }); #>
						</ul>
						<# } #>
					</div>
				</div>
			</div>
			<# } #>
			<?php
		}
	}