<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Social extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_social';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Social', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-social-icons';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-social' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_social',
				[
					'label' => __( 'Social', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'view',
				[
					'label'   => __( 'View', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::HIDDEN,
					'default' => 'traditional',
				]
			);
			$this->add_control(
				'social_style',
				[
					'label'   => __( 'Type', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'power_social_icon_hidetext',
					'options' => [
						'power_social_icon_hidetext' => __( 'Hide Text', 'power_elements_elementor' ),
						'power_social_icon_showtext' => __( 'Hide Icon', 'power_elements_elementor' ),
						'power_social_icon_notext'   => __( 'Only Icon', 'power_elements_elementor' )
					],
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'text',
				[
					'label'       => __( 'Text', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Facebook', 'power_elements_elementor' ),
					'placeholder' => __( 'Facebook', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'link',
				[
					'label'       => __( 'Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'default'     => [
						'url' => '#',
					],
				]
			);
			$repeater->add_control(
				'icon',
				[
					'label'       => __( 'Icon', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::ICON,
					'label_block' => true,
					'default'     => '',
				]
			);
			$this->add_control(
				'social_list',
				[
					'label'       => '',
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'text' => __( 'Social Item #1', 'power_elements_elementor' ),
							'icon' => 'fa fa-facebook',
						],
						[
							'text' => __( 'Social Item #2', 'power_elements_elementor' ),
							'icon' => 'fa fa-twitter',
						],
						[
							'text' => __( 'Social Item #3', 'power_elements_elementor' ),
							'icon' => 'fa fa-instagram',
						],
					],
					'title_field' => '<i class="{{ icon }}" aria-hidden="true"></i>{{{ text }}}',
				]
			);
			
			$this->add_responsive_control(
				'align',
				[
					'label'   => __( 'Alignment', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::CHOOSE,
					'options' => [
						'text-left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'text-center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'text-right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'text-justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'default' => '',
				]
			);
			
			$this->add_control(
				'social_css_id',
				[
					'label'       => __( 'Social ID', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => 'power-element-social-id',
					'title'       => __( 'Add your custom id WITHOUT the Pound key. e.g: my-id', 'power_elements_elementor' ),
					'label_block' => false,
					'description' => __( 'Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows <code>A-z 0-9</code> & underscore chars without spaces.', 'power_elements_elementor' ),
					'separator'   => 'before',
				
				]
			);
			
			$this->end_controls_section();
			$this->start_controls_section(
				'section_style',
				[
					'label' => __( 'Social', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'content_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_social_icon',
				]
			);
			$this->add_control(
				'social_animation',
				[
					'label'   => __( 'Animation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'animated',
					'options' => [
						'animated'     => __( 'Animation', 'power_elements_elementor' ),
						'not-animated' => __( 'No Animation', 'power_elements_elementor' )
					],
				]
			);
			$this->add_control(
				'animation_types',
				[
					'label'     => __( 'Animation Types', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SELECT,
					'default'   => 'shake',
					'options'   => [
						'shake'      => __( 'Shake', 'power_elements_elementor' ),
						'bounce'     => __( 'Bounce', 'power_elements_elementor' ),
						'flash'      => __( 'Flash', 'power_elements_elementor' ),
						'pulse'      => __( 'Pulse', 'power_elements_elementor' ),
						'swing'      => __( 'Swing', 'power_elements_elementor' ),
						'tada'       => __( 'Tada', 'power_elements_elementor' ),
						'wobble'     => __( 'Wobble', 'power_elements_elementor' ),
						'jello'      => __( 'Jello', 'power_elements_elementor' ),
						'zoomIn'     => __( 'Zoom In', 'power_elements_elementor' ),
						'heartBeat'  => __( 'Heart Beat', 'power_elements_elementor' ),
						'fadeIn'     => __( 'Fade In', 'power_elements_elementor' ),
						'rubberBand' => __( 'Rubber Band', 'power_elements_elementor' ),
					],
					'condition' => [
						'social_animation' => 'animated',
					],
				]
			);
			
			$this->start_controls_tabs( 'tabs_social_style' );
			
			$this->start_controls_tab(
				'tab_style_social_normal',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			
			$this->add_control(
				'social_text_color',
				[
					'label'     => __( 'Text Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_social_icon' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'background_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon, {{WRAPPER}} .power_social_icon' => 'background-color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'icon_size',
				[
					'label'     => __( 'Icon Size', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', 'em' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 60,
							'step' => 1,
						],
						'em'  => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 24,
					],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon i, {{WRAPPER}} .power_social_icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'icon_color',
				[
					'label'     => __( 'Icon Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon i, {{WRAPPER}} .power_social_icon i' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->end_controls_tab();
			
			$this->start_controls_tab(
				'tab_style_social_hover',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			
			$this->add_control(
				'hover_text_color',
				[
					'label'     => __( 'Text Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon:hover, {{WRAPPER}} .power_social_icon:hover, {{WRAPPER}} a.power_social_icon:focus, {{WRAPPER}} .power_social_icon:focus' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'social_background_hover_color',
				[
					'label'     => __( 'Background Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon:hover, {{WRAPPER}} .power_social_icon:hover, {{WRAPPER}} a.power_social_icon:focus, {{WRAPPER}} .power_social_icon:focus' => 'background-color: {{VALUE}};',
					],
				]
			);
			
			$this->add_control(
				'social_icon_hover_color',
				[
					'label'     => __( 'Icon Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon:hover i, {{WRAPPER}} .power_social_icon:hover i' => 'color: {{VALUE}};',
					],
				]
			);
			
			$this->end_controls_tab();
			
			$this->end_controls_tabs();
			$this->add_control(
				'hr-i',
				[
					'type' => \Elementor\Controls_Manager::DIVIDER,
				]
			);
			$this->start_controls_tabs( 'tabs_social_border' );
			
			$this->start_controls_tab(
				'tab_border_social_normal',
				[
					'label' => __( 'Normal', 'power_elements_elementor' ),
				]
			);
			
			
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'      => 'border',
					'selector'  => '{{WRAPPER}} .power_social_icon',
					'separator' => 'before',
				]
			);
			
			$this->add_control(
				'border_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon, {{WRAPPER}} .power_social_icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'social_box_shadow',
					'selector' => '{{WRAPPER}} .power_social_icon',
				]
			);
			
			$this->add_responsive_control(
				'text_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon, {{WRAPPER}} .power_social_icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->add_responsive_control(
				'text_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon, {{WRAPPER}} .power_social_icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'tab_border_social_before',
				[
					'label' => __( 'Hover', 'power_elements_elementor' ),
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'      => 'border_hover',
					'selector'  => '{{WRAPPER}} .power_social_icon:hover',
					'separator' => 'before',
				]
			);
			
			$this->add_control(
				'hover_border_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon:hover, {{WRAPPER}} .power_social_icon:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'hover_social_box_shadow',
					'selector' => '{{WRAPPER}} .power_social_icon:hover',
				]
			);
			
			$this->add_responsive_control(
				'hover_text_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon:hover, {{WRAPPER}} a.power_social_icon:hover, {{WRAPPER}} .power_social:before, {{WRAPPER}} .power_social:before' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_tab();
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$this->add_render_attribute( 'wrapper', 'class', [
				'power_social_icon_wrapper',
				'power_social_icon_rounded',
				$settings['social_style'],
				$settings['align']
			] );
			
			$html ='<div '. $this->get_render_attribute_string( 'wrapper' ).'>';
			foreach ( $settings['social_list'] as $item ) {
			
			if ( ! empty( $settings['social_css_id'] ) ) {
				$this->add_render_attribute( 'social', 'id', $settings['social_css_id'] );
			}

				if ( $settings['social_style'] != 'power_social_icon_notext' ) {
					$social_text = '<span class="power_social_icon_name">' . Power_elements_Helper::power_kses( $item['text'] ) . '</span>';
				} else {
					$social_text = '';
				}
				$html .='<a href="'.$item['link']['url'].'" class="power_social_icon '.$settings['animation_types'].' '.$settings['social_animation'].'">
					'. $social_text .'
                    <span class="power_social_icon_class"><i class="'. esc_attr( $item['icon'] ).'"></i></span>
                </a>';

			}
			$html .='</div>';
			
			echo $html;
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#

            view.addRenderAttribute( 'wrapper', {
            'class': [ 'power_social_icon_wrapper', 'power_social_icon_rounded', settings.social_style,  settings.align]
            } );

            #>
            <div class="elementor-social-wrapper">
                <div {{{ view.getRenderAttributeString(
                'wrapper' ) }}}>
                <# _.each( settings.social_list, function( item, index ) { #>
                <a href="{{ item.link.url }}" class="power_social_icon {{ settings.animation_types }} {{ settings.social_animation }}">
                <# if ( settings.social_style!='power_social_icon_notext' ) { #>
                <span class="power_social_icon_name">{{{settings.text}}}</span>
                <# } #>
                <span class="power_social_icon_class"><i class="{{{settings.icon}}}"></i></span>
                </a>
                 <#
                } ); #>
            </div>
            </div>
			<?php
		}
	}