<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Slider extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_slider';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Slider', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-media-carousel';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-slider' ];
		}
		
		/**
		 * ADD Widget JS
		 */
		
		public function get_script_depends() {
			return [ 'power-slider-js' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'power_element_slider_content',
				[
					'label' => __( 'Slider Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'slider_style',
				[
					'label'   => __( 'Style', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'slider_type1',
					'options' => [
						'slider_type1' => __( 'Style 1', 'power_elements_elementor' ),
						'slider_type2' => __( 'Style 2', 'power_elements_elementor' )
					]
				]
			);
			$repeater = new \Elementor\Repeater();
			$repeater->add_control(
				'sub_heading',
				[
					'label'       => __( 'Slider Sub Heading', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Slider Sub Heading', 'power_elements_elementor' ),
					'placeholder' => __( '', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'heading',
				[
					'label'       => __( 'Heading', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Slider Heading', 'power_elements_elementor' ),
					'placeholder' => __( '', 'power_elements_elementor' ),
					'label_block' => true
				]
			);
			$repeater->add_control(
				'description_text',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'default'     => __( 'Slider Heading', 'power_elements_elementor' ),
					'placeholder' => __( '', 'power_elements_elementor' ),
				]
			);
			$repeater->add_control(
				'slider_image',
				[
					'label'   => __( 'Choose Slider BG Image', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			$repeater->add_control(
				'slider_inner_image',
				[
					'label'   => __( 'Choose Inner Image', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);

			$repeater->add_control(
				'button_show_hide',
				[
					'label'        => __( 'Button', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SWITCHER,
					'label_on'     => __( 'show', 'power_elements_elementor' ),
					'label_off'    => __( 'Hide', 'power_elements_elementor' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$repeater->add_control(
				'link',
				[
					'label'       => __( 'Button Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'default'     => [
						'url' => '#',
					],
					'condition'   => [
						'button_show_hide' => 'yes',
					],
				]
			);
			$repeater->add_control(
				'button_text',
				[
					'label'       => __( 'Button Text', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Button Text', 'power_elements_elementor' ),
					'placeholder' => __( 'Read More', 'power_elements_elementor' ),
					'condition'   => [
						'button_show_hide' => 'yes',
					]
				]
			);
			$repeater->add_control(
				'button_style',
				[
					'label'        => __( 'Type', 'power_elements_elementor' ),
					'type'         => \Elementor\Controls_Manager::SELECT,
					'default'      => 'power_slider_button effect_1',
					'options'      => [
						'power_slider_button effect_1'  => __( 'Effect 1', 'power_elements_elementor' ),
						'power_slider_button effect_2'  => __( 'Effect 2', 'power_elements_elementor' ),
						'power_slider_button effect_3'  => __( 'Effect 3', 'power_elements_elementor' ),
						'power_slider_button effect_4'  => __( 'Effect 4', 'power_elements_elementor' ),
						'power_slider_button effect_5'  => __( 'Effect 5', 'power_elements_elementor' ),
						'power_slider_button effect_6'  => __( 'Effect 6', 'power_elements_elementor' ),
						'power_slider_button effect_7'  => __( 'Effect 7', 'power_elements_elementor' ),
						'power_slider_button effect_8'  => __( 'Effect 8', 'power_elements_elementor' ),
						'power_slider_button effect_9'  => __( 'Effect 9', 'power_elements_elementor' ),
					]
				]
			);
			$this->add_control(
				'list',
				[
					'label'       => __( 'Slider List', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'list_title'   => __( 'Title #1', 'power_elements_elementor' ),
							'list_content' => __( 'Item content.', 'power_elements_elementor' ),
						],
						[
							'list_title'   => __( 'Title #2', 'power_elements_elementor' ),
							'list_content' => __( 'Item content.', 'power_elements_elementor' ),
						],
					],
					'title_field' => '{{{ sub_heading }}}',
				]
			);
			$this->add_control(
				'slider_inner_form',
				[
					'label'   => __( 'Form Shortcode', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::TEXTAREA,					
					'condition' => [
						'slider_style' => 'slider_type2',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * CAROUSEL SETTINGS
			 */
			$this->start_controls_section(
				'section_slider_carousel_settings',
				[
					'label' => __( 'Settings', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'slider_carousel_item',
				[
					'label'       => __( 'Desktop Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on destop and tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'slider_carousel_item_mobile',
				[
					'label'       => __( 'Mobile Item', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::NUMBER,
					'min'         => 1,
					'max'         => 100,
					'step'        => 1,
					'default'     => 1,
					'description' => __( 'Number of item shows on mobile and small tablet screen', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'slider_carousel_margin',
				[
					'label'   => __( 'Margin', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'slider_carousel_padding',
				[
					'label'   => __( 'Padding', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SLIDER,
					'range'   => [
						'min'  => 0,
						'max'  => 200,
						'step' => 5,
					],
					'default' => [
						'size' => 30,
					],
				]
			);
			$this->add_control(
				'slider_carousel_autoplay',
				[
					'label'   => __( 'Autoplay', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'slider_carousel_responsive_class',
				[
					'label'   => __( 'Responsive Class', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'slider_carousel_dots',
				[
					'label'   => __( 'Dots', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->add_control(
				'slider_carousel_nav',
				[
					'label'   => __( 'Navigation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'true',
					'options' => [
						'true'  => __( 'True', 'power_elements_elementor' ),
						'false' => __( 'False', 'power_elements_elementor' ),
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE
			 */
			$this->start_controls_section(
				'power_element_style',
				[
					'label' => __( 'Slider Content Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'slider_img_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_item',
				]
			);
			$this->add_control(
				'slider_img_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'slider_img_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 *
			 * DETAILS STYLE
			 */
			
			$this->start_controls_section(
				'power_element_alignment',
				[
					'label' => __( 'Content Alignment', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'slider_details_align',
				[
					'label'     => __( 'Alignment', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'selectors' => [
						'{{WRAPPER}} .slide-content' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * Slider Description Style
			 */
			$this->start_controls_section(
				'power_element_des_style',
				[
					'label' => __( 'Slider Description', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_slider_item .slide-content .description',
				]
			);
			$this->add_control(
				'des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_slider_item .slide-content .description' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'des_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_item .slide-content .description',
				]
			);
			$this->add_responsive_control(
				'des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item .slide-content .description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_item .slide-content .description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Sub Heading Style
			 */
			$this->start_controls_section(
				'power_element_sub_heading_style',
				[
					'label' => __( 'Sub Heading Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slide-content .sub_heading',
				]
			);
			$this->add_control(
				'sub_heading_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slide-content .sub_heading' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'sub_heading_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .slide-content .sub_heading',
				]
			);
			$this->add_responsive_control(
				'sub_heading_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slide-content .sub_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'sub_heading_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slide-content .sub_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * Heading Style
			 */
			$this->start_controls_section(
				'power_element_heading_style',
				[
					'label' => __( 'Heading Style', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'heading_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .slide-content .default_heading',
				]
			);
			$this->add_control(
				'heading_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .slide-content .default_heading' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'heading_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .slide-content .default_heading',
				]
			);
			$this->add_responsive_control(
				'heading_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slide-content .default_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'heading_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .slide-content .default_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			/*
			 * STYLE BUTTON
			 */
			$this->start_controls_section(
				'section_button_style',
				[
					'label' => __( 'Button', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'button_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_control(
				'button_hover_color',
				[
					'label'     => __( 'Hover Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button:hover' => 'color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'button_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);

			$this->add_control(
				'button_bg_color_label',
				[
					'label' => __( 'Background Color', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'button_bg_color',
					'label'    => __( 'Background Color', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button:after',
				]
			);

			$this->add_control(
				'button_bg_hover_color_label',
				[
					'label' => __( 'Background Hover Color', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::HEADING,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'button_bg_hover_color',
					'label'    => __( 'Background Hover Color', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button:before',
				]
			);
			
			$this->add_control(
				'button_bg_hidden_color',
				[
					'label'     => __( 'Hidden Bg Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .power_slider_button' => 'background-color: {{VALUE}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'button_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'button_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_slider_button',
				]
			);
			$this->add_control(
				'button_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'button_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button_group' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'button_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_slider_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			if ( $settings['list'] ) {
				
				$html = '<div class="'.esc_html( $settings['slider_style'] ).'"><div class="owl-carousel owl-theme slider_carousel power_slider" data-owl-carousel=\'{"items": ' . esc_attr( $settings['slider_carousel_item'] ) . ', "margin": ' . esc_attr( $settings['slider_carousel_margin']['size'] ) . ', "padding": ' . esc_attr( $settings['slider_carousel_padding']['size'] ) . ', "autoplay": ' . esc_attr( $settings['slider_carousel_autoplay'] ) . ', "responsiveClass": ' . esc_attr( $settings['slider_carousel_responsive_class'] ) . ', "dots": ' . esc_attr( $settings['slider_carousel_dots'] ) . ', "nav": ' . esc_attr( $settings['slider_carousel_nav'] ) . ',"responsive":{ "0" :{ "items": ' . esc_attr( $settings['slider_carousel_item_mobile'] ) . ' }, "540" :{ "items": ' . esc_attr( $settings['slider_carousel_item_mobile'] ) . ' }, "860" :{ "items" : ' . esc_attr( $settings['slider_carousel_item'] ) . ' } , "1000":{ "items" : ' . esc_attr( $settings['slider_carousel_item'] ) . ' }}}\'>';
				foreach ( $settings['list'] as $item ) {

					if ( $item['button_show_hide']=='yes' ) {
					if ( ! empty( $item['link'] ) ) {
						$this->add_render_attribute( 'button', 'href', $item['link'] );
					}

					


					$button_html = '<div class="power_slider_button_group">
                                        <a ' . $this->get_render_attribute_string( 'button' ) . ' class="power_slider_button '.esc_attr( $item['button_style'] ).'">
                                            <span class="button_value">' . Power_elements_Helper::power_kses( $item['button_text'] ) . '</span>
                                        </a>
                                    </div>';
				} else {
					$button_html = '';
				}
					if ( $item['slider_image']['id'] != '' ) {
						$img = 'style="background-image:url(' . esc_url( $item['slider_image']['url'] ) . ')"';
					} else {
						$img = '';
					}
					
					if ( $item['description_text'] != '' ) {
						$description = Power_elements_Helper::power_kses( $item['description_text'] ) ;
					} else {
						$description = '';
					}

					if ( $settings['slider_style'] == 'slider_type1' ) {
						if ( is_array($item['slider_inner_image']) && $item['slider_inner_image']['url'] != '' ) {
							$slider_inner_image = '<div class="col-lg-5">
								<div class="slide-content-img">
									<img src="' . esc_url( $item['slider_inner_image']['url'] ) . '" alt="' . Power_elements_Helper::power_kses( $item['heading'] ) . '">
								</div>
							</div>';
						} else {
							$slider_inner_image = '';
						}
					} else {
						
							$slider_inner_form_html = '<div class="col-lg-5">
								<div class="slide-content-img">
									' . do_shortcode( $item['slider_inner_form'] ) . '
								</div>
							</div>';
						
					}



					if ( $settings['slider_style'] == 'slider_type2' ) {
						$html .= '<div class="item">
                        <div class="power_slider_item" ' . $img . '>
                            <div class="container">
                                <div class="row">
                                    <div class="col-lg-7">
                                        <div class="slide-content">
                                            <h1 class="default_heading">' . esc_html( $item['heading'] ) . '</h1>
                                            <h2 class="sub_heading">' . esc_html( $item['sub_heading'] ) . '</h2>
                                            <h4 class="description">' . $description . '</h4>
                                            ' . $button_html . '
                                        </div>
                                    </div>

                                    '.$slider_inner_form_html.'
                                </div>
                            </div>
                        </div>
                    </div>';
					} else {
						$html .= '<div class="item">
                        <div class="power_slider_item" ' . $img . '>
                            <div class="container">
                                <div class="row">
                                    <div class="col-lg-7">
                                        <div class="slide-content">
                                            <h4 class="description">' . $description . '</h4>
                                            <h2 class="sub_heading">' . esc_html( $item['sub_heading'] ) . '</h2>
                                            <h1 class="default_heading">' . esc_html( $item['heading'] ) . '</h1>
                                            ' . $button_html . '
                                        </div>
                                    </div>

                                    '.$slider_inner_image.'
                                </div>
                            </div>
                        </div>
                    </div>';
					}
				}
				$html .= '</div></div>';
				
				echo $html;
			} else {
				echo esc_html__( 'Something wrong', 'power_elements_elementor' );
			}
			
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}