<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Progressbar extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_progress';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Progressbar', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-skill-bar';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-progressbar' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_progress_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'progress_style',
				[
					'label'   => __( 'Style', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'power_progress_type1',
					'options' => [
						'power_progress_type1' => __( 'Style 1', 'power_elements_elementor' ),
						'power_progress_type2' => __( 'Style 2', 'power_elements_elementor' ),
					]
				]
			);
			$this->add_control(
				'progress_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Marketing', 'power_elements_elementor' ),
					'placeholder' => __( 'Title', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'progress_value',
				[
					'label'       => __( 'Value', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( '65%', 'power_elements_elementor' ),
					'placeholder' => __( 'Value%.', 'power_elements_elementor' ),
					'description' => __( 'Value is numerical, must include % sign.', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'progress_striped',
				[
					'label'   => __( 'Striped', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => 'striped',
					'options' => [
						'striped'    => __( 'Striped', 'power_elements_elementor' ),
						'no_striped' => __( 'No-striped', 'power_elements_elementor' ),
					]
				]
			);
			$this->add_control(
				'progress_striped_anime',
				[
					'label'     => __( 'Striped Animation', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::SELECT,
					'default'   => 'right_anim',
					'options'   => [
						'right_anim' => __( 'Right Anime', 'power_elements_elementor' ),
						'left_anim'  => __( 'Left Anime', 'power_elements_elementor' ),
					],
					'condition' => [
						'progress_striped' => 'striped',
					]
				]
			);
			$this->add_control(
				'progress_animation',
				[
					'label'   => __( 'Animation', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => '',
					'options' => [
						'bounce'     => __( 'Bounce', 'power_elements_elementor' ),
						'flash'      => __( 'Flash', 'power_elements_elementor' ),
						'pulse'      => __( 'Pulse', 'power_elements_elementor' ),
						'shake'      => __( 'Shake', 'power_elements_elementor' ),
						'swing'      => __( 'Swing', 'power_elements_elementor' ),
						'tada'       => __( 'Tada', 'power_elements_elementor' ),
						'wobble'     => __( 'Wobble', 'power_elements_elementor' ),
						'jello'      => __( 'Jello', 'power_elements_elementor' ),
						'zoomIn'     => __( 'Zoom In', 'power_elements_elementor' ),
						'fadeIn'     => __( 'Fade In', 'power_elements_elementor' ),
						'heartBeat'  => __( 'Heart Beat', 'power_elements_elementor' ),
						'rubberBand' => __( 'Rubber Band', 'power_elements_elementor' ),
					]
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR
			 */
			$this->start_controls_section(
				'power_element_progress_wrapper_style',
				[
					'label' => __( 'Progressbar', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'progress_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_content' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'progress_height',
				[
					'label' => __( 'Height', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 80,
							'step' => 1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_content' => 'height: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'progress_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_progress_content',
				]
			);
			$this->add_control(
				'progress_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR INNER
			 */
			$this->start_controls_section(
				'power_element_progress_inner_style',
				[
					'label' => __( 'Inner', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'progress_inner_height',
				[
					'label' => __( 'Height', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_content .power_progress_bar' => 'height: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Background::get_type(),
				[
					'name'     => 'progress_inner_bg_color',
					'label'    => __( 'Background', 'power_elements_elementor' ),
					'types'    => [ 'classic', 'gradient'],
					'selector' => '{{WRAPPER}} .power_progress_content .power_progress_bar',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'progress_inner_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_progress_content .power_progress_bar',
				]
			);
			$this->add_control(
				'progress_inner_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_bar' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_inner_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_bar' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_inner_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_bar' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR TEXT
			 */
			$this->start_controls_section(
				'power_element_progress_text_style',
				[
					'label' => __( 'Text', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'progress_text_color',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_progress_content .power_progress_text',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'progress_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_progress_content .power_progress_text',
				]
			);
			$this->add_control(
				'progress_text_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_content .power_progress_text' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'progress_text_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_text_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_text_top',
				[
					'label' => __( 'Top', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => -100,
							'max' => 100,
							'step' => 1,
						],
						'%' => [
							'min' => -100,
							'max' => 100,
						],
					],
					'condition' => [
						'progress_style' => 'power_progress_type1',
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_content .power_progress_text' => 'top: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->end_controls_section();
			/*
			 * STYLE PROGRESSBAR VALUE
			 */
			$this->start_controls_section(
				'power_element_progress_value_style',
				[
					'label' => __( 'Value', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'progress_value_color',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_progress_content .power_progress_value',
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'progress_value_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_progress_content .power_progress_value',
				]
			);
			$this->add_control(
				'progress_value_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_content .power_progress_value' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'progress_value_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_progress_content .power_progress_value' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'progress_value_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_value' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_value_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_progress_content .power_progress_value' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'progress_value_top',
				[
					'label' => __( 'Top', 'power_elements_elementor' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'range' => [
						'px' => [
							'min' => -100,
							'max' => 100,
							'step' => 1,
						],
						'%' => [
							'min' => -100,
							'max' => 100,
						],
					],
					'condition' => [
						'progress_style' => 'power_progress_type1',
					],
					'selectors' => [
						'{{WRAPPER}} .power_progress_content .power_progress_value' => 'top: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();
			
			$animation = $settings['progress_animation']!=''? ' '.$settings['progress_animation'].' animated' : '';
			$value = $settings['progress_value']!=''? $settings['progress_value'] : '0';
			
			echo '<div class="power_progress_content '.esc_attr($settings['progress_style']).' '.esc_attr($animation).'">
                <div class="power_progress_bar '.esc_attr($settings['progress_striped']).' '.esc_attr($settings['progress_striped_anime']).'" style="max-width: '.esc_attr($value).';">
                    <span class="power_progress_text">'.Power_elements_Helper::power_kses($settings['progress_title']).'</span>
                    <span class="power_progress_value">'.Power_elements_Helper::power_kses($value).'</span>
                </div>
            </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
			<div class="power_progress_content {{settings.progress_style}} {{settings.progress_animation}} animated">
				<div class="power_progress_bar {{settings.progress_striped}} {{settings.progress_striped_anime}}" style="max-width: {{settings.progress_value}};">
					<span class="power_progress_text">{{{settings.progress_title}}}</span>
					<span class="power_progress_value">{{{settings.progress_value}}}</span>
				</div>
			</div>
			<?php
		}
	}