<?php
	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class Power_Elements_Imagebox extends \Elementor\Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_element_imagebox';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Image Box', 'power_elements_elementor' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-image-box';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		/**
		 * ADD Widget CSS
		 */
		
		public function get_style_depends() {
			return [ 'power-element-imagebox' ];
		}
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_progress_content',
				[
					'label' => __( 'Content', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'imagebox_image',
				[
					'label'   => __( 'Choose Image', 'power_elements_elementor' ),
					'type'    => \Elementor\Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
				]
			);
			
			$this->add_group_control(
				\Elementor\Group_Control_Image_Size::get_type(),
				[
					'name'      => 'imagebox_thumbnail',
					// Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
					'default'   => 'full',
					'separator' => 'none',
				]
			);
			$this->add_control(
				'imagebox_link',
				[
					'label'       => __( 'Link', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::URL,
					'placeholder' => __( 'https://your-link.com', 'power_elements_elementor' ),
					'separator'   => 'before',
					'description' => __( 'NO need leave empty.', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'imagebox_title',
				[
					'label'       => __( 'Title', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXT,
					'default'     => __( 'Heading', 'power_elements_elementor' ),
					'placeholder' => __( 'Title', 'power_elements_elementor' ),
				]
			);
			$this->add_control(
				'imagebox_description',
				[
					'label'       => __( 'Description', 'power_elements_elementor' ),
					'type'        => \Elementor\Controls_Manager::TEXTAREA,
					'default'     => __( '', 'power_elements_elementor' ),
					'placeholder' => __( 'Description.', 'power_elements_elementor' ),
					'description' => __( 'No need leave empty', 'power_elements_elementor' ),
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'section_imagebox_alignment',
				[
					'label' => __( 'Alignment', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'imagebox_align',
				[
					'label'     => __( 'Alignment', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::CHOOSE,
					'options'   => [
						'left'   => [
							'title' => __( 'Left', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center' => [
							'title' => __( 'Center', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'right'  => [
							'title' => __( 'Right', 'power_elements_elementor' ),
							'icon'  => 'fa fa-align-right',
						]
					],
					'selectors' => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_img' => 'text-align: {{VALUE}}',
						'{{WRAPPER}} .power_imagebox_content'             => 'text-align: {{VALUE}}'
					],
				]
			);
			$this->end_controls_section();
			/*
			 * IMAGE BOX
			 */
			$this->start_controls_section(
				'section_imagebox',
				[
					'label' => __( 'Imagebox', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'imagebox_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_imagebox' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'imagebox_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox',
				]
			);
			$this->add_control(
				'imagebox_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'imagebox_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox',
				]
			);
			$this->add_control(
				'imagebox_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'imagebox_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * IMAGE BOX IMAGE
			 */
			$this->start_controls_section(
				'section_imagebox_image',
				[
					'label' => __( 'Image', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'imagebox_image_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_img img',
				]
			);
			$this->add_control(
				'imagebox_image_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_img img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'imagebox_image_box_shadow',
					'label' => __( 'Box Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_img img',
				]
			);
			$this->end_controls_section();
			/*
			 * IMAGE BOX CONTENT BOX
			 */
			$this->start_controls_section(
				'section_imagebox_contentbox',
				[
					'label' => __( 'Content Box', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_control(
				'imagebox_contentbox_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'imagebox_contentbox_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'imagebox_contentbox_bg_color',
				[
					'label'     => __( 'Background Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_content' => 'background-color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Border::get_type(),
				[
					'name'     => 'imagebox_contentbox_border',
					'label'    => __( 'Border', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_content',
				]
			);
			$this->add_control(
				'imagebox_contentbox_radius',
				[
					'label'      => __( 'Border Radius', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * IMAGE BOX TITLE
			 */
			$this->start_controls_section(
				'section_imagebox_title',
				[
					'label' => __( 'Heading', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'imagebox_title_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_title',
				]
			);
			$this->add_control(
				'imagebox_title_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_title' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'imagebox_text_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_title',
				]
			);
			$this->add_control(
				'imagebox_title_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'imagebox_title_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			/*
			 * IMAGE BOX DESCRIPTION
			 */
			$this->start_controls_section(
				'section_imagebox_des',
				[
					'label' => __( 'Description', 'power_elements_elementor' ),
					'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Typography::get_type(),
				[
					'name'     => 'imagebox_des_typography',
					'label'    => __( 'Typography', 'power_elements_elementor' ),
					'scheme'   => \Elementor\Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_description',
				]
			);
			$this->add_control(
				'imagebox_des_color',
				[
					'label'     => __( 'Color', 'power_elements_elementor' ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_description' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				\Elementor\Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'imagebox_des_shadow',
					'label'    => __( 'Text Shadow', 'power_elements_elementor' ),
					'selector' => '{{WRAPPER}} .power_imagebox .power_imagebox_description',
				]
			);
			$this->add_control(
				'imagebox_des_margin',
				[
					'label'      => __( 'Margin', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'imagebox_des_padding',
				[
					'label'      => __( 'Padding', 'power_elements_elementor' ),
					'type'       => \Elementor\Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} .power_imagebox .power_imagebox_description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings   = $this->get_settings_for_display();
			$image_html = \Elementor\Group_Control_Image_Size::get_attachment_image_html( $settings, 'imagebox_thumbnail', 'imagebox_image' );
			if ( ! empty( $settings['imagebox_link']['url'] ) ) {
				$this->add_render_attribute( 'link', 'href', $settings['link']['url'] );
				
				if ( $settings['link']['is_external'] ) {
					$this->add_render_attribute( 'link', 'target', '_blank' );
				}
				
				if ( ! empty( $settings['link']['nofollow'] ) ) {
					$this->add_render_attribute( 'link', 'rel', 'nofollow' );
				}
				$image_html = '<div class="power_imagebox_img"><a ' . $this->get_render_attribute_string( 'link' ) . '>' . $image_html . '</a></div>';
			}else{
			    $image_html ='<div class="power_imagebox_img">'.$image_html.'</div>';
            }
			$description = ! empty( $settings['imagebox_description'] ) ? '<p class="power_imagebox_description">' . Power_elements_Helper::power_kses( $settings['imagebox_description'] ) . '</p>' : '';
			echo '<div class="power_imagebox">
                ' . $image_html . '
                <div class="power_imagebox_content">
            	  <h3 class="power_imagebox_title">' . Power_elements_Helper::power_kses( $settings['imagebox_title'] ) . '</h3>
            	  ' . $description . '
                </div>
            </div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {
			?>
            <#
            if ( settings.imagebox_image.url ) {
            var image = {
            id: settings.imagebox_image.id,
            url: settings.imagebox_image.url,
            size: settings.imagebox_thumbnail_size,
            dimension: settings.imagebox_thumbnail_custom_dimension,
            model: view.getEditModel()
            };
            var html ='';
            var image_url = elementor.imagesManager.getImageUrl( image );

            var imageHtml = '<img src="' + image_url + '"/>';

            if ( settings.imagebox_link.url ) {
            imageHtml = '<a href="' + settings.imagebox_link.url + '">' + imageHtml + '</a>';
            }

            html +='<div class="power_imagebox_img">' + imageHtml + '</div>';
            }
            #>
            <div class="power_imagebox">
                {{{html}}}
                <div class="power_imagebox_content">
                    <h3 class="power_imagebox_title">{{{settings.imagebox_title}}}</h3>
                    <p class="power_imagebox_description">{{{settings.imagebox_description}}}</p>
                </div>
            </div>
			<?php
		}
	}